// Chinese Document Editor module for Overleaf Chrome Extension
// 中文文档编辑器：行编辑、同步、滚动等功能

// Live Translation Page Handlers
let currentProjectId = null;

// 记录当前高亮的行号（用于持续高亮）
let currentHighlightedLine = null;

// 文件切换版本控制 - 用于防止竞态条件
let fileLoadVersion = 0;

// 定时保存检查器
let autoSaveCheckInterval = null;
let lastSavedContent = null;  // 上次保存的内容，用于脏检查
let autoSavePausedUntil = 0;  // 自动保存暂停到此时间戳

// 注意：回环防止现在通过 source 标记实现，不再需要标志和时间戳

// 字号设置
const DEFAULT_FONT_SIZE = 14;
const MIN_FONT_SIZE = 10;
const MAX_FONT_SIZE = 24;
let currentFontSize = DEFAULT_FONT_SIZE;

/**
 * 初始化字号控制
 */
function initFontSizeControl() {
    // 从 localStorage 读取保存的字号
    const savedFontSize = localStorage.getItem('zh-editor-font-size');
    if (savedFontSize) {
        currentFontSize = parseInt(savedFontSize, 10);
        if (isNaN(currentFontSize) || currentFontSize < MIN_FONT_SIZE || currentFontSize > MAX_FONT_SIZE) {
            currentFontSize = DEFAULT_FONT_SIZE;
        }
    }

    // 应用字号
    applyFontSize();
    updateFontSizeDisplay();

    // 绑定按钮事件
    const decreaseBtn = document.getElementById('font-size-decrease');
    const increaseBtn = document.getElementById('font-size-increase');

    if (decreaseBtn) {
        decreaseBtn.addEventListener('click', () => {
            if (currentFontSize > MIN_FONT_SIZE) {
                currentFontSize--;
                applyFontSize();
                updateFontSizeDisplay();
                saveFontSize();
            }
        });
    }

    if (increaseBtn) {
        increaseBtn.addEventListener('click', () => {
            if (currentFontSize < MAX_FONT_SIZE) {
                currentFontSize++;
                applyFontSize();
                updateFontSizeDisplay();
                saveFontSize();
            }
        });
    }
}

/**
 * 应用字号到编辑器
 */
function applyFontSize() {
    const editor = document.getElementById('zh-line-editor');
    if (editor) {
        // 直接设置所有 .line-content 的字号（因为 CSS 中有固定字号会覆盖父元素）
        const lineContents = editor.querySelectorAll('.line-content');
        lineContents.forEach(el => {
            el.style.fontSize = `${currentFontSize}px`;
        });
        // 同时设置编辑器的字号，用于新创建的行
        editor.style.setProperty('--zh-editor-font-size', `${currentFontSize}px`);
    }
}

/**
 * 更新字号显示
 */
function updateFontSizeDisplay() {
    const display = document.getElementById('font-size-display');
    if (display) {
        display.textContent = currentFontSize;
    }
}

/**
 * 保存字号到 localStorage
 */
function saveFontSize() {
    localStorage.setItem('zh-editor-font-size', currentFontSize.toString());
}

/**
 * 滚动中文文档预览到指定行（只在编辑器内部滚动）
 * @param {number} lineNumber - 目标行号 (从1开始)
 * @param {boolean} persistent - 是否持续高亮（默认 true）
 */
function scrollZhDocumentToLine(lineNumber, persistent = true) {
    const editorDiv = document.getElementById('zh-line-editor');
    if (!editorDiv) return;

    // 如果是同一行，不需要重新处理
    if (persistent && currentHighlightedLine === lineNumber) {
        return;
    }

    // 移除之前高亮的行
    if (currentHighlightedLine !== null) {
        const previousLineRow = editorDiv.querySelector(`.line-row[data-line="${currentHighlightedLine}"]`);
        if (previousLineRow) {
            previousLineRow.classList.remove('highlight-line');
        }
    }

    // 找到对应行的元素
    const lineRowElement = editorDiv.querySelector(`.line-row[data-line="${lineNumber}"]`);
    if (lineRowElement) {
        // 计算目标位置，让目标行居中显示
        const editorRect = editorDiv.getBoundingClientRect();
        const rowRect = lineRowElement.getBoundingClientRect();
        const scrollTop = editorDiv.scrollTop + (rowRect.top - editorRect.top) - (editorRect.height / 2) + (rowRect.height / 2);

        editorDiv.scrollTo({
            top: Math.max(0, scrollTop),
            behavior: 'smooth'
        });

        // 添加高亮
        lineRowElement.classList.add('highlight-line');

        if (persistent) {
            // 持续高亮：记录当前高亮行号
            currentHighlightedLine = lineNumber;
        } else {
            // 短暂高亮：1秒后移除
            setTimeout(() => {
                lineRowElement.classList.remove('highlight-line');
            }, 1000);
        }
    }
}

/**
 * 刷新单行翻译 - 从Overleaf获取最新英文内容并翻译更新
 * @param {number} lineNumber - 行号 (从1开始)
 * @param {HTMLElement} lineNumElement - 行号元素
 */
async function refreshSingleLine(lineNumber, lineNumElement) {
    console.log('[Sidepanel] Refreshing line:', lineNumber);

    // 防止重复点击
    if (lineNumElement.classList.contains('loading')) {
        return;
    }

    // 设置加载状态
    lineNumElement.classList.add('loading');

    try {
        // 1. 从 Overleaf 获取最新的英文内容
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!tab) {
            throw new Error('无法获取当前标签页');
        }

        const englishContent = await new Promise((resolve, reject) => {
            chrome.tabs.sendMessage(tab.id, { type: 'GET_DOCUMENT_CONTENT' }, (response) => {
                if (chrome.runtime.lastError) {
                    reject(new Error(chrome.runtime.lastError.message));
                } else if (response && response.content) {
                    resolve(response.content);
                } else {
                    reject(new Error('无法获取编辑器内容'));
                }
            });
        });

        // 2. 获取指定行的英文内容
        const englishLines = englishContent.split('\n');
        if (lineNumber < 1 || lineNumber > englishLines.length) {
            throw new Error(`行号 ${lineNumber} 超出范围 (1-${englishLines.length})`);
        }

        const englishLineText = englishLines[lineNumber - 1];
        console.log('[Sidepanel] English line to translate:', englishLineText);

        // 3. 如果是空行或太短，不需要翻译
        if (!doubaoTranslator.shouldTranslateLine(englishLineText)) {
            // 直接更新为原文
            updateLineContent(lineNumber, englishLineText);
            showSuccess(`第 ${lineNumber} 行已更新`);
            return;
        }

        // 4. 调用翻译 API
        const model = document.getElementById('live-translation-model-select').value;
        // 根据模型类型获取对应的思考参数
        const reasoningEffort = await getSelectedReasoningEffort();

        // 检查是否使用免费模型
        if (model === 'microsoft-translator') {
            // 使用微软翻译
            const translatedText = await microsoftTranslator.translate(englishLineText, 'en', 'zh-Hans');
            updateLineContent(lineNumber, translatedText);
        } else {
            // 使用豆包翻译 API
            const requestBody = {
                lines: { [lineNumber]: englishLineText }
            };
            if (model) {
                requestBody.model = model;
            }
            if (reasoningEffort) {
                requestBody.reasoning_effort = reasoningEffort;
            }

            const response = await doubaoTranslator.fetchWithTimeout(`${BACKEND_API_URL}/api/translate/en-to-zh`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                credentials: 'include',
                body: JSON.stringify(requestBody)
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.detail || `HTTP ${response.status}`);
            }

            const result = await response.json();

            // 5. 更新翻译结果
            if (result.translated_lines && result.translated_lines[lineNumber]) {
                const translatedText = result.translated_lines[lineNumber];
                updateLineContent(lineNumber, translatedText);

                // 保存到本地存储
                const projectId = await getCurrentProjectId();
                if (projectId && currentFileName) {
                    await localDocStorage.updateLines(projectId, currentFileName, { [lineNumber]: translatedText });
                }
            } else {
                // 如果没有翻译结果，使用原文
                updateLineContent(lineNumber, englishLineText);
            }

            // 更新剩余 token 显示
            if (result.remaining_tokens !== undefined) {
                document.getElementById('user-tokens').textContent = result.remaining_tokens.toLocaleString();
            }
        }

        showSuccess(`第 ${lineNumber} 行翻译已更新`);

        // 清除该行的 modified 状态和确认按钮（因为已经通过行号刷新更新了）
        clearLineModifiedState(lineNumber);

    } catch (error) {
        console.error('[Sidepanel] Error refreshing line:', error);
        showAIModelError(`更新第 ${lineNumber} 行失败: ${error.message}`);
    } finally {
        // 恢复状态
        lineNumElement.classList.remove('loading');
    }
}

/**
 * 更新指定行的中文内容
 * @param {number} lineNumber - 行号 (从1开始)
 * @param {string} newContent - 新的内容
 */
function updateLineContent(lineNumber, newContent) {
    const editorDiv = document.getElementById('zh-line-editor');
    const lineContent = editorDiv.querySelector(`.line-content[data-line="${lineNumber}"]`);
    if (lineContent) {
        lineContent.textContent = newContent;
        // 高亮显示更新的行（短暂高亮，不影响持续高亮逻辑）
        const lineRow = lineContent.closest('.line-row');
        if (lineRow) {
            // 如果当前行不是持续高亮行，则短暂高亮
            if (currentHighlightedLine !== lineNumber) {
                lineRow.classList.add('highlight-line');
                setTimeout(() => {
                    // 只有在这个行不是当前持续高亮行时才移除
                    if (currentHighlightedLine !== lineNumber) {
                        lineRow.classList.remove('highlight-line');
                    }
                }, 1500);
            }
        }
        // 更新字符数
        const content = getZhContent();
        document.getElementById('zh-content-length').textContent = content.length.toLocaleString();
    }
}

/**
 * 给指定行添加 modified 状态和确认按钮
 * @param {number} lineNumber - 行号 (从1开始)
 */
function markLineAsModified(lineNumber) {
    const editorDiv = document.getElementById('zh-line-editor');
    const lineRow = editorDiv.querySelector(`.line-row[data-line="${lineNumber}"]`);
    if (!lineRow) return;

    if (!lineRow.classList.contains('modified')) {
        lineRow.classList.add('modified');

        // 添加确认翻译按钮（如果不存在）
        if (!lineRow.querySelector('.confirm-translate-btn')) {
            const lineContentDiv = lineRow.querySelector('.line-content');
            const confirmBtn = document.createElement('button');
            confirmBtn.className = 'confirm-translate-btn';
            confirmBtn.innerHTML = '<svg viewBox="0 0 24 24"><polyline points="20 6 9 17 4 12"></polyline></svg>';
            confirmBtn.title = '翻译并同步到 Overleaf';
            confirmBtn.addEventListener('click', async (e) => {
                e.stopPropagation();
                const lineNum = parseInt(lineContentDiv.dataset.line);
                await translateAndSyncToOverleaf(lineNum, lineContentDiv, confirmBtn);
            });
            lineRow.insertBefore(confirmBtn, lineRow.firstChild);
        }
    }
}

/**
 * 移除指定行的 modified 状态和确认按钮
 * @param {number} lineNumber - 行号 (从1开始)
 */
function clearLineModifiedState(lineNumber) {
    const editorDiv = document.getElementById('zh-line-editor');
    const lineRow = editorDiv.querySelector(`.line-row[data-line="${lineNumber}"]`);
    if (!lineRow) return;

    lineRow.classList.remove('modified');
    const confirmBtn = lineRow.querySelector('.confirm-translate-btn');
    if (confirmBtn) {
        confirmBtn.remove();
    }
}

/**
 * 获取当前项目ID
 */
async function getCurrentProjectId() {
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (tab && tab.url) {
            const match = tab.url.match(/\/project\/([a-f0-9]+)/);
            if (match) {
                return match[1];
            }
        }
    } catch (error) {
        console.error('[Sidepanel] Error getting project ID:', error);
    }
    return null;
}

function renderZhDocument(content, projectId = null, fileName = null) {
    // Debug: Check what content we receive
    const newlineCount = (content.match(/\n/g) || []).length;
    console.log('[Sidepanel] renderZhDocument called with', newlineCount, 'newlines, length:', content.length);

    const lines = content.split('\n');
    console.log('[Sidepanel] Split into', lines.length, 'lines');

    const editorDiv = document.getElementById('zh-line-editor');

    // 在编辑器上记录当前文档的 projectId 和 fileName
    // 这样自动保存时可以使用这些信息，而不是全局变量
    if (projectId && fileName) {
        editorDiv.dataset.projectId = projectId;
        editorDiv.dataset.fileName = fileName;
        console.log('[Sidepanel] Editor bound to:', projectId, fileName);
    }

    // Generate line rows - 刷新图标使用 SVG
    const refreshSvg = '<svg viewBox="0 0 24 24"><path d="M21 12a9 9 0 1 1-9-9c2.52 0 4.93 1 6.74 2.74L21 8"/><path d="M21 3v5h-5"/></svg>';
    let rowsHTML = '';
    for (let i = 0; i < lines.length; i++) {
        const lineContent = lines[i] || '';
        rowsHTML += `
            <div class="line-row" data-line="${i + 1}">
                <div class="line-number" data-line="${i + 1}" title="点击更新此行翻译"><span class="line-num-text">${i + 1}</span><span class="refresh-icon">${refreshSvg}</span></div>
                <div class="line-content" contenteditable="true" data-line="${i + 1}">${escapeHtml(lineContent)}</div>
            </div>
        `;
    }
    editorDiv.innerHTML = rowsHTML;

    // Add click handlers to both line numbers and line content
    const jumpToLineHandler = async (lineNumber) => {
        console.log('[Sidepanel] Line clicked:', lineNumber);
        try {
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            if (tab) {
                chrome.tabs.sendMessage(tab.id, {
                    type: 'JUMP_TO_LINE',
                    lineNumber: lineNumber
                }, (response) => {
                    if (chrome.runtime.lastError) {
                        console.error('[Sidepanel] Jump to line error:', chrome.runtime.lastError);
                    } else if (response && response.success) {
                        console.log('[Sidepanel] Successfully jumped to line:', lineNumber);
                    }
                });
            }
        } catch (error) {
            console.error('[Sidepanel] Error jumping to line:', error);
        }
    };

    // 点击行号时触发刷新翻译
    editorDiv.querySelectorAll('.line-number').forEach(lineNumDiv => {
        lineNumDiv.addEventListener('click', async (e) => {
            e.stopPropagation();
            const lineNumber = parseInt(lineNumDiv.dataset.line);
            await refreshSingleLine(lineNumber, lineNumDiv);
        });
    });

    editorDiv.querySelectorAll('.line-content').forEach(lineContentDiv => {
        bindLineContentEvents(lineContentDiv, editorDiv, jumpToLineHandler);
    });

    // Update character count
    document.getElementById('zh-content-length').textContent = content.length.toLocaleString();
}

// 绑定行内容的事件处理
function bindLineContentEvents(lineContentDiv, editorDiv, jumpToLineHandler) {
    lineContentDiv.addEventListener('click', async (e) => {
        // 在简略视图模式下，不触发跳转到 Overleaf，让用户可以正常编辑
        if (typeof isCompactMode !== 'undefined' && isCompactMode) {
            // 简略模式：只聚焦到编辑区域，不跳转
            return;
        }
        const lineNumber = parseInt(e.target.dataset.line);
        await jumpToLineHandler(lineNumber);
    });

    // Add keydown handler for Enter key and Backspace/Delete
    lineContentDiv.addEventListener('keydown', async (e) => {
        if (e.key === 'Enter') {
            e.preventDefault(); // Prevent default Enter behavior

            // Get current line
            const currentLineDiv = e.target;
            const currentLineRow = currentLineDiv.parentElement;
            const currentLineNumber = parseInt(currentLineDiv.dataset.line);
            const currentLineText = currentLineDiv.textContent;

            // Get cursor position
            const selection = window.getSelection();
            const range = selection.getRangeAt(0);

            // Split content at cursor position
            const beforeCursor = range.cloneRange();
            beforeCursor.selectNodeContents(currentLineDiv);
            beforeCursor.setEnd(range.startContainer, range.startOffset);
            const textBefore = beforeCursor.toString();

            const afterCursor = range.cloneRange();
            afterCursor.selectNodeContents(currentLineDiv);
            afterCursor.setStart(range.endContainer, range.endOffset);
            const textAfter = afterCursor.toString();

            // 判断是否在行首按回车（且当前行不为空）
            const isAtLineStart = textBefore === '' && currentLineText.trim() !== '';

            // Create new line row
            const newLineRow = document.createElement('div');
            newLineRow.className = 'line-row';

            const newLineNumber = document.createElement('div');
            newLineNumber.className = 'line-number';
            newLineNumber.title = '点击更新此行翻译';
            newLineNumber.innerHTML = '<span class="line-num-text"></span><span class="refresh-icon"><svg viewBox="0 0 24 24"><path d="M21 12a9 9 0 1 1-9-9c2.52 0 4.93 1 6.74 2.74L21 8"/><path d="M21 3v5h-5"/></svg></span>';

            const newLineContent = document.createElement('div');
            newLineContent.className = 'line-content';
            newLineContent.contentEditable = 'true';

            if (isAtLineStart) {
                // 在行首按回车：在当前行前面插入空行
                newLineContent.textContent = '';
                currentLineRow.parentNode.insertBefore(newLineRow, currentLineRow);

                // Renumber all lines
                newLineRow.appendChild(newLineNumber);
                newLineRow.appendChild(newLineContent);
                renumberAllLines(editorDiv);

                // Bind events to new line
                newLineNumber.addEventListener('click', async (evt) => {
                    evt.stopPropagation();
                    const lineNum = parseInt(newLineNumber.dataset.line);
                    await refreshSingleLine(lineNum, newLineNumber);
                });
                bindLineContentEvents(newLineContent, editorDiv, jumpToLineHandler);

                // 光标保持在原来的行（现在是下一行）
                currentLineDiv.focus();
                const newRange = document.createRange();
                newRange.setStart(currentLineDiv.childNodes[0] || currentLineDiv, 0);
                newRange.collapse(true);
                selection.removeAllRanges();
                selection.addRange(newRange);

                // Update character count
                updateZhCharCount();

                // 同步到 Overleaf：在当前行前面插入空行（即在 currentLineNumber - 1 之后插入）
                await syncInsertLineToOverleaf(currentLineNumber - 1);
            } else {
                // 正常情况：在当前行后面插入新行，内容为光标后的文字
                currentLineDiv.textContent = textBefore;
                newLineContent.textContent = textAfter;

                newLineRow.appendChild(newLineNumber);
                newLineRow.appendChild(newLineContent);

                // Insert new line after current line
                currentLineRow.parentNode.insertBefore(newLineRow, currentLineRow.nextSibling);

                // Renumber all lines
                renumberAllLines(editorDiv);

                // Bind events to new line
                newLineNumber.addEventListener('click', async (evt) => {
                    evt.stopPropagation();
                    const lineNum = parseInt(newLineNumber.dataset.line);
                    await refreshSingleLine(lineNum, newLineNumber);
                });
                bindLineContentEvents(newLineContent, editorDiv, jumpToLineHandler);

                // Move cursor to beginning of new line
                newLineContent.focus();
                const newRange = document.createRange();
                newRange.setStart(newLineContent.childNodes[0] || newLineContent, 0);
                newRange.collapse(true);
                selection.removeAllRanges();
                selection.addRange(newRange);

                // Update character count
                updateZhCharCount();

                // 同步到 Overleaf：在对应位置插入新行
                await syncInsertLineToOverleaf(currentLineNumber);
            }

        } else if (e.key === 'Backspace') {
            const currentLineDiv = e.target;
            const currentLineText = currentLineDiv.textContent;
            const currentLineRow = currentLineDiv.parentElement;
            const allLineRows = editorDiv.querySelectorAll('.line-row');
            const currentLineNumber = parseInt(currentLineDiv.dataset.line);

            // 检查光标是否在行首
            const selection = window.getSelection();
            const isAtLineStart = selection.anchorOffset === 0 && selection.isCollapsed;

            if (currentLineText.trim() === '') {
                // 当前行为空，删除当前行
                if (allLineRows.length > 1) {
                    e.preventDefault();

                    const prevLineRow = currentLineRow.previousElementSibling;
                    const nextLineRow = currentLineRow.nextElementSibling;

                    currentLineRow.remove();
                    renumberAllLines(editorDiv);

                    // 聚焦到上一行末尾或下一行开头
                    if (prevLineRow) {
                        const prevLineContent = prevLineRow.querySelector('.line-content');
                        prevLineContent.focus();
                        const sel = window.getSelection();
                        const range = document.createRange();
                        if (prevLineContent.childNodes.length > 0) {
                            const lastNode = prevLineContent.childNodes[prevLineContent.childNodes.length - 1];
                            range.setStart(lastNode, lastNode.length || 0);
                        } else {
                            range.setStart(prevLineContent, 0);
                        }
                        range.collapse(true);
                        sel.removeAllRanges();
                        sel.addRange(range);
                    } else if (nextLineRow) {
                        const nextLineContent = nextLineRow.querySelector('.line-content');
                        nextLineContent.focus();
                        const sel = window.getSelection();
                        const range = document.createRange();
                        range.setStart(nextLineContent.childNodes[0] || nextLineContent, 0);
                        range.collapse(true);
                        sel.removeAllRanges();
                        sel.addRange(range);
                    }

                    updateZhCharCount();
                    await syncDeleteLineToOverleaf(currentLineNumber);
                }
            } else if (isAtLineStart && currentLineNumber > 1) {
                // 当前行不为空，光标在行首，且不是第一行
                e.preventDefault();

                const prevLineRow = currentLineRow.previousElementSibling;
                if (!prevLineRow) return;

                const prevLineContent = prevLineRow.querySelector('.line-content');
                const prevLineText = prevLineContent.textContent;
                const prevLineNumber = currentLineNumber - 1;

                if (prevLineText.trim() === '') {
                    // 前一行为空，直接删除前一行
                    prevLineRow.remove();
                    renumberAllLines(editorDiv);
                    updateZhCharCount();
                    await syncDeleteLineToOverleaf(prevLineNumber);
                } else {
                    // 前一行不为空，将当前行合并到前一行末尾
                    const mergePosition = prevLineText.length; // 记录合并位置，用于设置光标
                    prevLineContent.textContent = prevLineText + currentLineText;

                    // 删除当前行
                    currentLineRow.remove();
                    renumberAllLines(editorDiv);

                    // 聚焦到合并位置
                    prevLineContent.focus();
                    const sel = window.getSelection();
                    const range = document.createRange();
                    if (prevLineContent.childNodes.length > 0) {
                        const textNode = prevLineContent.childNodes[0];
                        range.setStart(textNode, mergePosition);
                    } else {
                        range.setStart(prevLineContent, 0);
                    }
                    range.collapse(true);
                    sel.removeAllRanges();
                    sel.addRange(range);

                    updateZhCharCount();

                    // 同步到 Overleaf：合并行
                    await syncMergeLinesToOverleaf(prevLineNumber, currentLineNumber);
                }
            }
        } else if (e.key === 'Delete') {
            // Delete 键只处理当前行为空的情况
            const currentLineDiv = e.target;
            const currentLineText = currentLineDiv.textContent.trim();

            if (currentLineText === '') {
                const currentLineRow = currentLineDiv.parentElement;
                const allLineRows = editorDiv.querySelectorAll('.line-row');

                if (allLineRows.length > 1) {
                    e.preventDefault();

                    const currentLineNumber = parseInt(currentLineDiv.dataset.line);
                    const nextLineRow = currentLineRow.nextElementSibling;

                    currentLineRow.remove();
                    renumberAllLines(editorDiv);

                    if (nextLineRow) {
                        const nextLineContent = nextLineRow.querySelector('.line-content');
                        nextLineContent.focus();
                        const sel = window.getSelection();
                        const range = document.createRange();
                        range.setStart(nextLineContent.childNodes[0] || nextLineContent, 0);
                        range.collapse(true);
                        sel.removeAllRanges();
                        sel.addRange(range);
                    }

                    updateZhCharCount();
                    await syncDeleteLineToOverleaf(currentLineNumber);
                }
            }
        }
    });

    // 监听内容变化，标记为已修改
    lineContentDiv.addEventListener('input', () => {
        const lineRow = lineContentDiv.parentElement;

        // 标记行为已修改
        if (!lineRow.classList.contains('modified')) {
            lineRow.classList.add('modified');

            // 添加确认翻译按钮（如果不存在）
            if (!lineRow.querySelector('.confirm-translate-btn')) {
                const confirmBtn = document.createElement('button');
                confirmBtn.className = 'confirm-translate-btn';
                // 使用 SVG 勾选图标
                confirmBtn.innerHTML = '<svg viewBox="0 0 24 24"><polyline points="20 6 9 17 4 12"></polyline></svg>';
                confirmBtn.title = '翻译并同步到 Overleaf';
                confirmBtn.addEventListener('click', async (e) => {
                    e.stopPropagation();
                    const lineNumber = parseInt(lineContentDiv.dataset.line);
                    await translateAndSyncToOverleaf(lineNumber, lineContentDiv, confirmBtn);
                });
                // 插入到行的最前面（行号之前）
                lineRow.insertBefore(confirmBtn, lineRow.firstChild);
            }
        }
        // 注意：保存由 startAutoSaveChecker 定时检查处理，不需要在这里手动触发
    });
}

// 重新编号所有行
function renumberAllLines(editorDiv) {
    const allLineRows = editorDiv.querySelectorAll('.line-row');
    allLineRows.forEach((row, index) => {
        const lineNum = row.querySelector('.line-number');
        const lineNumText = lineNum.querySelector('.line-num-text');
        const lineContent = row.querySelector('.line-content');
        const lineIndex = index + 1;

        if (lineNumText) {
            lineNumText.textContent = lineIndex;
        } else {
            // 旧格式的行号，直接设置文本
            lineNum.textContent = lineIndex;
        }
        lineNum.dataset.line = lineIndex;
        row.dataset.line = lineIndex;
        if (lineContent) {
            lineContent.dataset.line = lineIndex;
        }
    });
}

// 更新中文字符数
function updateZhCharCount() {
    const content = getZhContent();
    document.getElementById('zh-content-length').textContent = content.length.toLocaleString();
}

// 同步插入行到 Overleaf（使用精确插入命令）
async function syncInsertLineToOverleaf(afterLineNumber) {
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (tab) {
            console.log(`[Sidepanel] Syncing Enter: inserting line after line ${afterLineNumber}`);

            chrome.tabs.sendMessage(tab.id, {
                type: 'INSERT_LINE',
                afterLineNumber: afterLineNumber,
                source: 'zh-editor'  // 标记来源，用于防止回环
            }, (response) => {
                if (!chrome.runtime.lastError && response && response.success) {
                    console.log(`[Sidepanel] Successfully synced line insertion to English doc`);
                } else {
                    console.log(`[Sidepanel] Failed to sync line insertion`);
                }
            });
        }
    } catch (error) {
        console.error('[Sidepanel] Error syncing Enter to English doc:', error);
    }
}

// 同步删除行到 Overleaf（使用精确删除命令）
async function syncDeleteLineToOverleaf(lineNumber) {
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (tab) {
            console.log(`[Sidepanel] Syncing Delete: removing line ${lineNumber}`);

            chrome.tabs.sendMessage(tab.id, {
                type: 'DELETE_LINE',
                lineNumber: lineNumber,
                source: 'zh-editor'  // 标记来源，用于防止回环
            }, (response) => {
                if (!chrome.runtime.lastError && response && response.success) {
                    console.log(`[Sidepanel] Successfully synced line deletion to English doc`);
                } else {
                    console.log(`[Sidepanel] Failed to sync line deletion`);
                }
            });
        }
    } catch (error) {
        console.error('[Sidepanel] Error syncing Delete to English doc:', error);
    }
}

// 同步合并行到 Overleaf（将 fromLineNumber 行合并到 toLineNumber 行末尾）
async function syncMergeLinesToOverleaf(toLineNumber, fromLineNumber) {
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (tab) {
            console.log(`[Sidepanel] Syncing Merge: merging line ${fromLineNumber} into line ${toLineNumber}`);

            chrome.tabs.sendMessage(tab.id, {
                type: 'MERGE_LINES',
                toLineNumber: toLineNumber,
                fromLineNumber: fromLineNumber,
                source: 'zh-editor'  // 标记来源，用于防止回环
            }, (response) => {
                if (!chrome.runtime.lastError && response && response.success) {
                    console.log(`[Sidepanel] Successfully synced line merge to English doc`);
                } else {
                    console.log(`[Sidepanel] Failed to sync line merge`);
                }
            });
        }
    } catch (error) {
        console.error('[Sidepanel] Error syncing Merge to English doc:', error);
    }
}

// 处理中间回车：插入新行并调用行号刷新功能翻译更新两行
// 注意：回环防止现在在 sidepanel.js 的 DOC_LINE_CHANGED 处理中通过 source 标记实现
async function handleMiddleInsert(prevLineNumber, currentLineNumber) {
    console.log(`[Sidepanel] Handling middle insert: prevLine=${prevLineNumber}, currentLine=${currentLineNumber}`);

    try {
        // 1. 在中文文档中插入一个新行
        const editorDiv = document.getElementById('zh-line-editor');
        if (!editorDiv) {
            console.error('[Sidepanel] zh-line-editor not found');
            return;
        }

        const allLineRows = editorDiv.querySelectorAll('.line-row');
        console.log(`[Sidepanel] Current zh doc has ${allLineRows.length} lines, inserting after line ${prevLineNumber}`);

        // 如果中文文档行数不够，跳过
        if (prevLineNumber > allLineRows.length) {
            console.log(`[Sidepanel] prevLineNumber ${prevLineNumber} > zh doc lines ${allLineRows.length}, skipping insert`);
            return;
        }

        // 找到插入位置（prevLineNumber 之后）
        const insertAfterRow = prevLineNumber > 0 && prevLineNumber <= allLineRows.length
            ? allLineRows[prevLineNumber - 1]
            : null;

        // 创建新行
        const newLineRow = document.createElement('div');
        newLineRow.className = 'line-row';
        newLineRow.dataset.line = currentLineNumber;

        const newLineNumber = document.createElement('div');
        newLineNumber.className = 'line-number';
        newLineNumber.title = '点击更新此行翻译';
        newLineNumber.dataset.line = currentLineNumber;
        newLineNumber.innerHTML = '<span class="line-num-text"></span><span class="refresh-icon"><svg viewBox="0 0 24 24"><path d="M21 12a9 9 0 1 1-9-9c2.52 0 4.93 1 6.74 2.74L21 8"/><path d="M21 3v5h-5"/></svg></span>';

        const newLineContent = document.createElement('div');
        newLineContent.className = 'line-content';
        newLineContent.contentEditable = 'true';
        newLineContent.dataset.line = currentLineNumber;
        newLineContent.textContent = '(加载中...)';

        newLineRow.appendChild(newLineNumber);
        newLineRow.appendChild(newLineContent);

        // 插入到正确位置
        if (insertAfterRow && insertAfterRow.nextSibling) {
            editorDiv.insertBefore(newLineRow, insertAfterRow.nextSibling);
        } else if (insertAfterRow) {
            editorDiv.appendChild(newLineRow);
        } else {
            editorDiv.insertBefore(newLineRow, editorDiv.firstChild);
        }
        console.log(`[Sidepanel] Inserted new line after row ${prevLineNumber}`);

        // 同时标记 prevLineNumber 行为加载中（因为它的内容也会被更新）
        if (insertAfterRow) {
            const prevLineContent = insertAfterRow.querySelector('.line-content');
            if (prevLineContent) {
                prevLineContent.dataset.originalContent = prevLineContent.textContent;
                prevLineContent.textContent = '(加载中...)';
            }
        }

        // 绑定新行的事件
        newLineNumber.addEventListener('click', async (evt) => {
            evt.stopPropagation();
            const lineNum = parseInt(newLineNumber.dataset.line);
            await refreshSingleLine(lineNum, newLineNumber);
        });

        const jumpHandler = async (lineNum) => {
            try {
                const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
                if (tab) {
                    chrome.tabs.sendMessage(tab.id, { type: 'JUMP_TO_LINE', lineNumber: lineNum });
                }
            } catch (e) {}
        };
        bindLineContentEvents(newLineContent, editorDiv, jumpHandler);

        // 重新编号所有行
        renumberAllLines(editorDiv);
        updateZhCharCount();

        // 2. 等待 Overleaf 编辑器状态稳定，然后强制刷新内容缓存
        await new Promise(resolve => setTimeout(resolve, 150));

        // 获取 tab 并使用 GET_FRESH_DOCUMENT_CONTENT 强制刷新缓存
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (tab) {
            // GET_FRESH_DOCUMENT_CONTENT 会从 injected.js 获取最新内容并更新缓存
            await new Promise((resolve) => {
                chrome.tabs.sendMessage(tab.id, { type: 'GET_FRESH_DOCUMENT_CONTENT' }, (response) => {
                    if (response && response.content) {
                        console.log(`[Sidepanel] Cache refreshed with ${response.content.split('\\n').length} lines`);
                    }
                    resolve();
                });
            });
        }

        // 3. 获取更新后的行号元素并调用 refreshSingleLine（它会获取最新英文内容并翻译）
        const updatedAllLineRows = editorDiv.querySelectorAll('.line-row');
        console.log(`[Sidepanel] Zh doc now has ${updatedAllLineRows.length} lines, refreshing lines ${prevLineNumber} and ${currentLineNumber}`);

        const prevLineNumEl = prevLineNumber > 0 && prevLineNumber <= updatedAllLineRows.length
            ? updatedAllLineRows[prevLineNumber - 1].querySelector('.line-number')
            : null;
        const currentLineNumEl = currentLineNumber > 0 && currentLineNumber <= updatedAllLineRows.length
            ? updatedAllLineRows[currentLineNumber - 1].querySelector('.line-number')
            : null;

        // 并行调用 refreshSingleLine - 它会自动获取最新英文内容并翻译
        const translatePromises = [];
        if (prevLineNumEl) {
            console.log(`[Sidepanel] Calling refreshSingleLine for line ${prevLineNumber}`);
            translatePromises.push(refreshSingleLine(prevLineNumber, prevLineNumEl));
        }
        if (currentLineNumEl) {
            console.log(`[Sidepanel] Calling refreshSingleLine for line ${currentLineNumber}`);
            translatePromises.push(refreshSingleLine(currentLineNumber, currentLineNumEl));
        }

        await Promise.all(translatePromises);
        console.log(`[Sidepanel] Middle insert complete - both lines refreshed and translated`);
        // 注意：保存由 startAutoSaveChecker 定时检查处理
    } catch (error) {
        console.error('[Sidepanel] Error handling middle insert:', error);
    }
}

// 在中文文档中插入新行（从 Overleaf 同步过来）
// 注意：回环防止现在在 sidepanel.js 的 DOC_LINE_CHANGED 处理中通过 source 标记实现
function insertZhDocumentLines(afterLineNumber, count = 1) {
    const editorDiv = document.getElementById('zh-line-editor');
    if (!editorDiv) {
        return;
    }

    const allLineRows = editorDiv.querySelectorAll('.line-row');

    // 找到插入位置
    let insertAfterRow = null;
    if (afterLineNumber > 0 && afterLineNumber <= allLineRows.length) {
        insertAfterRow = allLineRows[afterLineNumber - 1];
    }

    // 创建新行
    for (let i = 0; i < count; i++) {
        const newLineRow = document.createElement('div');
        newLineRow.className = 'line-row';

        const newLineNumber = document.createElement('div');
        newLineNumber.className = 'line-number';
        newLineNumber.title = '点击更新此行翻译';
        newLineNumber.innerHTML = '<span class="line-num-text"></span><span class="refresh-icon"><svg viewBox="0 0 24 24"><path d="M21 12a9 9 0 1 1-9-9c2.52 0 4.93 1 6.74 2.74L21 8"/><path d="M21 3v5h-5"/></svg></span>';

        const newLineContent = document.createElement('div');
        newLineContent.className = 'line-content';
        newLineContent.contentEditable = 'true';
        newLineContent.textContent = '';

        newLineRow.appendChild(newLineNumber);
        newLineRow.appendChild(newLineContent);

        // 插入到正确位置
        if (insertAfterRow && insertAfterRow.nextSibling) {
            editorDiv.insertBefore(newLineRow, insertAfterRow.nextSibling);
            insertAfterRow = newLineRow;
        } else if (insertAfterRow) {
            editorDiv.appendChild(newLineRow);
            insertAfterRow = newLineRow;
        } else {
            editorDiv.insertBefore(newLineRow, editorDiv.firstChild);
            insertAfterRow = newLineRow;
        }

        // 绑定事件
        newLineNumber.addEventListener('click', async (evt) => {
            evt.stopPropagation();
            const lineNum = parseInt(newLineNumber.dataset.line);
            await refreshSingleLine(lineNum, newLineNumber);
        });

        const jumpHandler = async (lineNum) => {
            try {
                const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
                if (tab) {
                    chrome.tabs.sendMessage(tab.id, {
                        type: 'JUMP_TO_LINE',
                        lineNumber: lineNum
                    });
                }
            } catch (e) {}
        };

        bindLineContentEvents(newLineContent, editorDiv, jumpHandler);
    }

    // 重新编号所有行
    renumberAllLines(editorDiv);
    updateZhCharCount();

    console.log(`[Sidepanel] Inserted ${count} lines after line ${afterLineNumber}`);
    // 注意：保存由 startAutoSaveChecker 定时检查处理
}

// 合并行：将 fromLine 的内容合并到 toLine，然后删除 fromLine
// 注意：回环防止现在在 sidepanel.js 的 DOC_LINE_CHANGED 处理中通过 source 标记实现
function mergeZhDocumentLines(toLineNumber, fromLineNumber) {
    console.log(`[Sidepanel] Merging line ${fromLineNumber} into line ${toLineNumber}`);

    const editorDiv = document.getElementById('zh-line-editor');
    if (!editorDiv) {
        return;
    }

    const allLineRows = editorDiv.querySelectorAll('.line-row');

    const toLineIndex = toLineNumber - 1;
    const fromLineIndex = fromLineNumber - 1;

    if (toLineIndex < 0 || toLineIndex >= allLineRows.length ||
        fromLineIndex < 0 || fromLineIndex >= allLineRows.length) {
        console.log('[Sidepanel] Invalid line numbers for merge');
        return;
    }

    const toLineRow = allLineRows[toLineIndex];
    const fromLineRow = allLineRows[fromLineIndex];

    const toLineContent = toLineRow.querySelector('.line-content');
    const fromLineContent = fromLineRow.querySelector('.line-content');

    if (!toLineContent || !fromLineContent) {
        return;
    }

    // 合并内容
    const existingText = toLineContent.textContent || '';
    const appendText = fromLineContent.textContent || '';
    toLineContent.textContent = existingText + appendText;

    // 删除被合并的行
    fromLineRow.remove();

    // 重新编号所有行
    renumberAllLines(editorDiv);
    updateZhCharCount();

    console.log(`[Sidepanel] Merged line ${fromLineNumber} into line ${toLineNumber}`);
    // 注意：保存由 startAutoSaveChecker 定时检查处理
}

// 在中文文档中删除行（从 Overleaf 同步过来）
// 注意：回环防止现在在 sidepanel.js 的 DOC_LINE_CHANGED 处理中通过 source 标记实现
function deleteZhDocumentLines(atLineNumber, count = 1) {
    const editorDiv = document.getElementById('zh-line-editor');
    if (!editorDiv) {
        return;
    }

    const allLineRows = editorDiv.querySelectorAll('.line-row');

    // 确保至少保留一行
    if (allLineRows.length <= count) {
        console.log('[Sidepanel] Cannot delete all lines, keeping at least one');
        return;
    }

    // 删除指定位置的行
    for (let i = 0; i < count; i++) {
        const lineIndex = atLineNumber - 1;
        const currentRows = editorDiv.querySelectorAll('.line-row');

        if (lineIndex >= 0 && lineIndex < currentRows.length) {
            currentRows[lineIndex].remove();
        }
    }

    // 重新编号所有行
    renumberAllLines(editorDiv);
    updateZhCharCount();

    console.log(`[Sidepanel] Deleted ${count} lines at line ${atLineNumber}`);
    // 注意：保存由 startAutoSaveChecker 定时检查处理
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function getZhContent() {
    const editorDiv = document.getElementById('zh-line-editor');
    const lines = Array.from(editorDiv.querySelectorAll('.line-content'));
    return lines.map(line => line.textContent).join('\n');
}

/**
 * 暂停自动保存一段时间（用于文件切换时等待文档加载完成）
 * @param {number} seconds - 暂停秒数
 */
function pauseAutoSave(seconds) {
    autoSavePausedUntil = Date.now() + seconds * 1000;
    console.log('[Sidepanel] Auto-save paused for', seconds, 'seconds');
}

/**
 * 启动自动保存检查器
 * 每秒检查中文文档内容是否有变化，如果有变化则保存
 * 关键：从编辑器元素上读取 projectId 和 fileName，确保内容和文件名匹配
 */
function startAutoSaveChecker() {
    // 先停止已有的检查器
    stopAutoSaveChecker();

    console.log('[Sidepanel] Starting auto-save checker');

    autoSaveCheckInterval = setInterval(async () => {
        // 检查是否在暂停期间
        if (Date.now() < autoSavePausedUntil) {
            return;
        }

        // 从编辑器元素上读取 projectId 和 fileName
        // 这样可以确保保存的内容和文件名是匹配的
        const editorDiv = document.getElementById('zh-line-editor');
        const editorProjectId = editorDiv?.dataset?.projectId;
        const editorFileName = editorDiv?.dataset?.fileName;

        // 检查编辑器是否绑定了有效的文件
        if (!editorProjectId || !editorFileName) {
            return;
        }

        // 获取当前编辑器内容
        const editorContent = getZhContent();

        // 如果内容与上次保存的相同，不需要保存
        if (editorContent === lastSavedContent) {
            return;
        }

        // 内容有变化，保存
        console.log('[Sidepanel] Content changed, saving to:', editorFileName);

        try {
            await localDocStorage.save(editorProjectId, editorFileName, editorContent);

            // 保存成功后更新 lastSavedContent
            lastSavedContent = editorContent;
            console.log('[Sidepanel] Auto-saved zh document:', editorFileName);

            // 更新状态显示
            const now = new Date().toLocaleTimeString();
            document.getElementById('zh-preview-status').textContent = `(已保存: ${now})`;
        } catch (error) {
            console.error('[Sidepanel] Auto-save failed:', error);
        }
    }, 4000);  // 每 4 秒检查一次
}

/**
 * 停止自动保存检查器
 */
function stopAutoSaveChecker() {
    if (autoSaveCheckInterval) {
        clearInterval(autoSaveCheckInterval);
        autoSaveCheckInterval = null;
        console.log('[Sidepanel] Stopped auto-save checker');
    }
}

/**
 * 在 Overleaf 编辑器中显示 diff 预览悬浮框
 * @param {number} lineNumber - 行号
 * @param {string} originalText - 原始英文内容
 * @param {string} updatedText - 更新后的英文内容
 * @param {HTMLElement} lineRow - 行元素
 * @param {HTMLElement} confirmBtn - 确认按钮
 * @param {object} tab - Chrome tab 对象
 * @param {number} remainingTokens - 剩余点数（可选）
 * @returns {Promise<boolean>} - 用户是否确认
 */
async function showDiffPreviewInOverleaf(lineNumber, originalText, updatedText, lineRow, confirmBtn, tab, remainingTokens) {
    console.log(`[Sidepanel] Showing diff popup in Overleaf for line ${lineNumber}`);

    try {
        // 发送消息到 content.js，让它在 Overleaf 页面中显示悬浮框
        const response = await new Promise((resolve, reject) => {
            chrome.tabs.sendMessage(tab.id, {
                type: 'SHOW_DIFF_POPUP',
                lineNumber: lineNumber,
                originalText: originalText,
                updatedText: updatedText
            }, (response) => {
                if (chrome.runtime.lastError) {
                    reject(new Error(chrome.runtime.lastError.message));
                } else {
                    resolve(response);
                }
            });
        });

        console.log('[Sidepanel] Diff popup response:', response);

        if (response && response.confirmed) {
            // 用户确认了更新
            if (response.success) {
                // 更新用户点数显示
                if (remainingTokens !== undefined) {
                    updateUserTokensDisplay(remainingTokens);
                }

                // 清除修改状态
                lineRow.classList.remove('modified');
                if (confirmBtn) confirmBtn.remove();

                // 短暂高亮显示成功
                lineRow.classList.add('highlight-line');
                setTimeout(() => {
                    lineRow.classList.remove('highlight-line');
                }, 1000);

                showSuccess(`第 ${lineNumber} 行已更新并同步到 Overleaf`);
                return true;
            } else {
                showError(`同步失败: ${response.error || '未知错误'}`);
                return false;
            }
        } else {
            // 用户取消了
            console.log('[Sidepanel] User cancelled diff popup');
            // 恢复按钮状态
            if (confirmBtn) {
                confirmBtn.classList.remove('loading');
                confirmBtn.innerHTML = '<svg viewBox="0 0 24 24"><polyline points="20 6 9 17 4 12"></polyline></svg>';
            }
            return false;
        }
    } catch (error) {
        console.error('[Sidepanel] Error showing diff popup:', error);
        showError(`显示预览失败: ${error.message}`);
        // 恢复按钮状态
        if (confirmBtn) {
            confirmBtn.classList.remove('loading');
            confirmBtn.innerHTML = '<svg viewBox="0 0 24 24"><polyline points="20 6 9 17 4 12"></polyline></svg>';
        }
        return false;
    }
}

// 翻译中文行并同步到 Overleaf
async function translateAndSyncToOverleaf(lineNumber, lineContentDiv, confirmBtn) {
    const lineRow = lineContentDiv.parentElement;
    const zhContent = lineContentDiv.textContent;

    if (!zhContent || !zhContent.trim()) {
        // 空行，直接清除修改状态
        lineRow.classList.remove('modified');
        if (confirmBtn) confirmBtn.remove();
        return;
    }

    console.log(`[Sidepanel] Updating line ${lineNumber} to English:`, zhContent.substring(0, 50));

    // 设置加载状态
    if (confirmBtn) {
        confirmBtn.classList.add('loading');
        // 显示旋转的加载图标
        confirmBtn.innerHTML = '<svg viewBox="0 0 24 24"><path d="M21 12a9 9 0 1 1-6.219-8.56"></path></svg>';
    }

    try {
        // 1. 先从 Overleaf 获取当前英文内容
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!tab) {
            throw new Error('无法获取当前标签页');
        }

        const englishContent = await new Promise((resolve, reject) => {
            chrome.tabs.sendMessage(tab.id, { type: 'GET_DOCUMENT_CONTENT' }, (response) => {
                if (chrome.runtime.lastError) {
                    reject(new Error(chrome.runtime.lastError.message));
                } else if (response && response.content) {
                    resolve(response.content);
                } else {
                    reject(new Error('无法获取编辑器内容'));
                }
            });
        });

        // 获取指定行的英文内容
        const englishLines = englishContent.split('\n');
        if (lineNumber < 1 || lineNumber > englishLines.length) {
            throw new Error(`行号 ${lineNumber} 超出范围 (1-${englishLines.length})`);
        }

        const enContent = englishLines[lineNumber - 1];
        console.log(`[Sidepanel] Current English content:`, enContent.substring(0, 50));

        // 2. 获取当前选择的模型和思考程度
        const modelSelect = document.getElementById('live-translation-model-select');
        const model = modelSelect ? modelSelect.value : 'doubao-translation';
        // 根据模型类型获取对应的思考参数
        const reasoningEffort = await getSelectedReasoningEffort();

        let updatedText;
        let remainingTokens = null;

        // 3. 根据模型选择翻译方式
        if (model === 'microsoft-translator') {
            // 免费模型：直接调用微软翻译 API 进行中译英
            console.log('[Sidepanel] Using Microsoft Translator for zh-to-en');
            updatedText = await microsoftTranslator.translate(zhContent, 'zh-Hans', 'en');
            console.log('[Sidepanel] Microsoft translated:', updatedText.substring(0, 50));
        } else {
            // 付费模型：调用后端 API
            const response = await doubaoTranslator.fetchWithTimeout(`${BACKEND_API_URL}/api/translate/zh-update-en`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                credentials: 'include',
                body: JSON.stringify({
                    lines: {
                        [lineNumber]: {
                            zh: zhContent,
                            en: enContent
                        }
                    },
                    model: model,
                    reasoning_effort: reasoningEffort
                })
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.detail || `更新失败: ${response.status}`);
            }

            const result = await response.json();
            console.log('[Sidepanel] Update result:', result);

            if (!result.success || !result.updated_lines) {
                throw new Error('更新结果无效');
            }

            updatedText = result.updated_lines[lineNumber];
            if (updatedText === undefined) {
                throw new Error('未找到更新结果');
            }

            // 处理返回值可能是对象的情况（如 {en: "...", zh: "..."} 格式）
            if (typeof updatedText === 'object' && updatedText !== null) {
                // 尝试从对象中提取英文内容
                updatedText = updatedText.en || updatedText.content || updatedText.text || JSON.stringify(updatedText);
            }

            // 确保 updatedText 是字符串
            if (typeof updatedText !== 'string') {
                updatedText = String(updatedText);
            }

            remainingTokens = result.remaining_tokens;
        }

        console.log(`[Sidepanel] Updated English:`, updatedText.substring(0, 50));

        // 4. 在 Overleaf 编辑器中显示 diff 预览悬浮框，让用户确认
        await showDiffPreviewInOverleaf(
            lineNumber,
            enContent,
            updatedText,
            lineRow,
            confirmBtn,
            tab,
            remainingTokens
        );

        // 注意：保存由 startAutoSaveChecker 定时检查处理

    } catch (error) {
        console.error('[Sidepanel] Update error:', error);
        showAIModelError(`更新失败: ${error.message}`);

        // 恢复按钮状态
        if (confirmBtn) {
            confirmBtn.classList.remove('loading');
            // 恢复勾选图标
            confirmBtn.innerHTML = '<svg viewBox="0 0 24 24"><polyline points="20 6 9 17 4 12"></polyline></svg>';
        }
    }
}

// 更新用户点数显示
function updateUserTokensDisplay(tokens) {
    const tokenDisplay = document.getElementById('user-tokens');
    if (tokenDisplay) {
        tokenDisplay.textContent = tokens;
    }
}

/**
 * 加载中文文档
 * 逻辑：
 * 1. 检查本地存储是否已有该文档
 * 2. 如果有，直接加载显示
 * 3. 如果没有，从 Overleaf 获取英文内容，复制保存，然后显示
 * 4. 加载完成后启动自动保存检查器
 *
 * @param {number} tabId - 可选，Chrome tab ID，用于获取英文内容
 */
async function fetchZhDocument(tabId = null) {
    // 递增版本号，标记新的加载请求
    fileLoadVersion++;
    const thisLoadVersion = fileLoadVersion;

    // 停止之前的自动保存检查器
    stopAutoSaveChecker();

    // 重置 lastSavedContent，防止与旧文件内容比较
    lastSavedContent = null;

    // 捕获当前的上下文变量，避免在异步操作期间被修改
    const projectId = currentProjectId;
    const fileName = currentFileName;

    console.log('[Sidepanel] fetchZhDocument called, version:', thisLoadVersion, 'projectId:', projectId, 'fileName:', fileName);

    if (!projectId || !fileName) {
        console.log('[Sidepanel] Missing projectId or fileName, skipping load');
        return;
    }

    // 检查是否为支持边写边译的文件类型（只有 .tex 和 .txt）
    if (!isLiveTranslationFile(fileName)) {
        console.log('[Sidepanel] Not a live translation file, skipping:', fileName);
        if (thisLoadVersion !== fileLoadVersion) {
            console.log('[Sidepanel] Load version mismatch (not text file check), skipping render');
            return;
        }
        renderZhDocument('');
        document.getElementById('zh-preview-status').textContent = '(不支持此文件类型)';
        document.getElementById('full-translate-btn').disabled = true;
        document.getElementById('reverse-translate-btn').disabled = true;
        document.getElementById('reload-english-btn').disabled = true;
        return;
    }

    try {
        // 步骤1：检查本地存储是否已有该文档
        console.log('[Sidepanel] Checking local storage for:', fileName);
        const doc = await localDocStorage.load(projectId, fileName);

        // 异步操作后检查版本是否仍然有效
        if (thisLoadVersion !== fileLoadVersion) {
            console.log('[Sidepanel] Load version mismatch after storage check, aborting. thisVersion:', thisLoadVersion, 'currentVersion:', fileLoadVersion);
            return;
        }

        let loadedContent = null;

        if (doc && doc.content !== undefined && doc.content !== null) {
            // 步骤2：本地已有文档，直接加载显示（不获取英文内容）
            console.log('[Sidepanel] Found existing document in local storage, loading directly');
            loadedContent = doc.content;
            renderZhDocument(loadedContent, projectId, fileName);
            const savedTime = doc.savedAt ? new Date(doc.savedAt).toLocaleTimeString() : '未知';
            document.getElementById('zh-preview-status').textContent = `(最后更新: ${savedTime})`;
            console.log('[Sidepanel] Loaded document from local storage, lines:', doc.lineCount);
        } else {
            // 步骤3：本地没有文档，需要从 Overleaf 获取英文内容并复制
            console.log('[Sidepanel] No existing document, need to fetch English content from Overleaf');

            // 获取 tab ID（如果没有传入）
            let currentTabId = tabId;
            if (!currentTabId) {
                const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
                currentTabId = tab?.id;
            }

            if (!currentTabId) {
                console.log('[Sidepanel] Cannot get tab ID, creating empty document');
                loadedContent = '';
                renderZhDocument(loadedContent, projectId, fileName);
                document.getElementById('zh-preview-status').textContent = '(无内容)';
            } else {
                // 从 Overleaf 获取英文内容
                const englishContent = await new Promise((resolve) => {
                    chrome.tabs.sendMessage(currentTabId, { type: 'GET_FRESH_DOCUMENT_CONTENT' }, (response) => {
                        if (chrome.runtime.lastError) {
                            console.log('[Sidepanel] Error getting English content:', chrome.runtime.lastError.message);
                            resolve(null);
                            return;
                        }
                        resolve(response?.content || null);
                    });
                });

                // 检查版本
                if (thisLoadVersion !== fileLoadVersion) {
                    console.log('[Sidepanel] Load version mismatch after fetching English content, aborting');
                    return;
                }

                if (englishContent) {
                    const newlineCount = (englishContent.match(/\n/g) || []).length;
                    console.log('[Sidepanel] Got English content:', newlineCount, 'lines, length:', englishContent.length);

                    // 更新全局的 currentContent
                    currentContent = englishContent;
                    document.getElementById('content-length').textContent = englishContent.length.toLocaleString();

                    // 首次创建，保存到本地存储
                    await localDocStorage.save(projectId, fileName, englishContent);

                    // 保存后再次检查版本
                    if (thisLoadVersion !== fileLoadVersion) {
                        console.log('[Sidepanel] Load version mismatch after initial save, aborting render');
                        return;
                    }

                    loadedContent = englishContent;
                    renderZhDocument(loadedContent, projectId, fileName);
                    document.getElementById('zh-preview-status').textContent = '(已复制原文档，点击翻译开始)';
                    console.log('[Sidepanel] Created and saved new document from English content');
                } else {
                    // 没有英文内容可复制
                    console.log('[Sidepanel] No English content available');
                    loadedContent = '';
                    renderZhDocument(loadedContent, projectId, fileName);
                    document.getElementById('zh-preview-status').textContent = '(无内容)';
                }
            }
        }

        // 步骤4：设置 lastSavedContent 并启动自动保存检查器
        if (thisLoadVersion === fileLoadVersion) {
            lastSavedContent = loadedContent;
            startAutoSaveChecker();
        }

    } catch (error) {
        console.error('[Sidepanel] Error in fetchZhDocument:', error);
    }
}
