// Compact View module for Overleaf Chrome Extension
// 简略视图：将边写边译模块通过 iframe 嵌入到 Overleaf 页面下方

// 简略视图状态
let isCompactViewActive = false;

/**
 * 切换到简略视图
 * 向 content.js 发送消息，在 Overleaf 页面创建简略视图面板
 * 同时关闭当前的 sidepanel
 */
async function toggleCompactView() {
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!tab) {
            showError('无法获取当前标签页');
            return;
        }

        // 检查当前是否在氛围写作模式
        // isAmbientWritingMode 在 ambient-writing.js 中定义
        const currentMode = (typeof isAmbientWritingMode !== 'undefined' && isAmbientWritingMode)
            ? 'ambient-writing'
            : 'translation';

        console.log('[CompactView] Opening compact view with mode:', currentMode);

        // 打开简略视图（通过 iframe 嵌入 sidepanel）
        chrome.tabs.sendMessage(tab.id, {
            type: 'OPEN_COMPACT_VIEW',
            mode: currentMode
        }, (response) => {
            if (chrome.runtime.lastError) {
                console.error('[Sidepanel] Error opening compact view:', chrome.runtime.lastError);
                showError('无法打开简略视图，请确保在 Overleaf 页面');
                return;
            }

            if (response && response.success) {
                // 简略视图打开成功后，关闭当前 sidepanel
                // 通过发送消息给 background.js 来关闭 sidepanel
                window.close();
            }
        });
    } catch (error) {
        console.error('[Sidepanel] Error toggling compact view:', error);
        showError('切换简略视图失败: ' + error.message);
    }
}

/**
 * 更新简略视图按钮状态
 */
function updateCompactViewButton() {
    const btn = document.getElementById('compact-view-btn');
    if (!btn) return;

    if (isCompactViewActive) {
        btn.textContent = '退出简略';
        btn.classList.remove('bg-blue-600', 'hover:bg-blue-700');
        btn.classList.add('bg-orange-600', 'hover:bg-orange-700');
        btn.title = '关闭简略视图';
    } else {
        btn.textContent = '简略视图';
        btn.classList.remove('bg-orange-600', 'hover:bg-orange-700');
        btn.classList.add('bg-blue-600', 'hover:bg-blue-700');
        btn.title = '将中文文档移至Overleaf编辑器下方';
    }
}

/**
 * 处理来自简略视图的消息
 */
function handleCompactViewMessage(request) {
    switch (request.action) {
        case 'VIEW_CLOSED':
            // 简略视图被关闭（用户点击了关闭按钮）
            isCompactViewActive = false;
            updateCompactViewButton();
            break;
    }
}

/**
 * 检查简略视图状态
 */
async function checkCompactViewStatus() {
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!tab) return;

        chrome.tabs.sendMessage(tab.id, {
            type: 'IS_COMPACT_VIEW_OPEN'
        }, (response) => {
            if (chrome.runtime.lastError) {
                return;
            }

            if (response) {
                isCompactViewActive = response.isOpen;
                updateCompactViewButton();
            }
        });
    } catch (error) {
        console.error('[Sidepanel] Error checking compact view status:', error);
    }
}

/**
 * 初始化简略视图按钮事件
 */
function initCompactViewButton() {
    const btn = document.getElementById('compact-view-btn');
    if (btn) {
        btn.addEventListener('click', toggleCompactView);
    }

    // 检查当前简略视图状态
    checkCompactViewStatus();
}

// 导出函数供其他模块使用
if (typeof window !== 'undefined') {
    window.toggleCompactView = toggleCompactView;
    window.handleCompactViewMessage = handleCompactViewMessage;
    window.initCompactViewButton = initCompactViewButton;
    window.isCompactViewActive = () => isCompactViewActive;
}
