// Ambient Writing module for Overleaf Chrome Extension
// 氛围写作：LLM 对话助手，支持代码引用

// AI 模型错误的建议提示
const AI_MODEL_ERROR_HINT_AMBIENT = '\n💡 建议：尝试更高级的模型或更高的思考程度';

// =================== 状态管理 ===================
let isAmbientWritingMode = false;
let chatHistory = []; // 对话历史
let isDrawMode = false; // 绘图模式状态

/**
 * 获取当前文件名（从全局变量或返回默认值）
 */
function getCurrentFileName() {
    // currentFileName 在 sidepanel.js 中定义为全局变量
    if (typeof currentFileName !== 'undefined') {
        return currentFileName;
    }
    return 'unknown';
}

// =================== 模式切换 ===================

/**
 * 更新简略视图按钮的文字，根据当前是氛围写作还是边写边译
 */
function updateCompactViewButtonText() {
    const btnText = document.getElementById('compact-view-btn-text');
    if (!btnText) return;

    if (isAmbientWritingMode) {
        btnText.textContent = '氛围写作简略';
    } else {
        btnText.textContent = '边写边译简略';
    }
}

/**
 * 切换到氛围写作模式
 */
function enterAmbientWritingMode() {
    console.log('[AmbientWriting] Entering ambient writing mode');
    isAmbientWritingMode = true;

    // 隐藏中文文档预览，显示 LLM 对话框
    const zhPreviewWrapper = document.getElementById('zh-preview-wrapper');
    const llmChatWrapper = document.getElementById('llm-chat-wrapper');

    // 检查是否是简略模式（isCompactMode 在 sidepanel.js 中定义）
    const inCompactMode = typeof isCompactMode !== 'undefined' && isCompactMode;

    if (zhPreviewWrapper) {
        zhPreviewWrapper.style.display = 'none';
    }
    if (llmChatWrapper) {
        llmChatWrapper.classList.add('active');
        if (inCompactMode) {
            // 简略模式下需要设置正确的样式
            llmChatWrapper.style.display = 'flex';
            llmChatWrapper.style.flex = '1';
            llmChatWrapper.style.margin = '0';
            llmChatWrapper.style.borderRadius = '0';
            llmChatWrapper.style.boxShadow = 'none';
        }
    }

    // 更新按钮状态 - 激活状态使用蓝色
    const ambientBtn = document.getElementById('ambient-writing-btn');
    if (ambientBtn) {
        ambientBtn.classList.remove('bg-white', 'hover:bg-slate-50', 'text-slate-700', 'border-slate-200');
        ambientBtn.classList.add('bg-blue-500', 'hover:bg-blue-600', 'text-white', 'border-blue-500');
        ambientBtn.innerHTML = `
            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
            </svg>
            写作中...
        `;
    }

    // 更新简略视图按钮文字
    updateCompactViewButtonText();

    // 聚焦到输入框
    setTimeout(() => {
        const input = document.getElementById('llm-chat-input');
        if (input) input.focus();
    }, 100);
}

/**
 * 退出氛围写作模式
 */
function exitAmbientWritingMode() {
    console.log('[AmbientWriting] Exiting ambient writing mode');
    isAmbientWritingMode = false;

    // 显示中文文档预览，隐藏 LLM 对话框
    const zhPreviewWrapper = document.getElementById('zh-preview-wrapper');
    const llmChatWrapper = document.getElementById('llm-chat-wrapper');

    // 检查是否是简略模式（isCompactMode 在 sidepanel.js 中定义）
    const inCompactMode = typeof isCompactMode !== 'undefined' && isCompactMode;

    if (zhPreviewWrapper) {
        zhPreviewWrapper.style.display = 'flex';
        if (inCompactMode) {
            // 简略模式下需要设置正确的样式
            zhPreviewWrapper.style.flex = '1';
            zhPreviewWrapper.style.margin = '0';
            zhPreviewWrapper.style.borderRadius = '0';
            zhPreviewWrapper.style.border = 'none';
            zhPreviewWrapper.style.boxShadow = 'none';
        }
    }
    if (llmChatWrapper) {
        llmChatWrapper.classList.remove('active');
        if (inCompactMode) {
            // 简略模式下隐藏氛围写作容器
            llmChatWrapper.style.display = 'none';
        }
    }

    // 恢复按钮状态 - 默认白色
    const ambientBtn = document.getElementById('ambient-writing-btn');
    if (ambientBtn) {
        ambientBtn.classList.remove('bg-blue-500', 'hover:bg-blue-600', 'text-white', 'border-blue-500');
        ambientBtn.classList.add('bg-white', 'hover:bg-slate-50', 'text-slate-700', 'border-slate-200');
        ambientBtn.innerHTML = `
            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"></path>
            </svg>
            氛围写作
        `;
    }

    // 更新简略视图按钮文字
    updateCompactViewButtonText();
}

/**
 * 切换氛围写作模式
 */
function toggleAmbientWritingMode() {
    if (isAmbientWritingMode) {
        exitAmbientWritingMode();
    } else {
        enterAmbientWritingMode();
    }
}

// =================== 对话功能 ===================

/**
 * 添加消息到对话框
 * @param {string} content - 消息内容
 * @param {string} role - 角色: 'user', 'assistant', 'system'
 */
function addChatMessage(content, role) {
    const messagesContainer = document.getElementById('llm-chat-messages');
    if (!messagesContainer) return;

    const messageDiv = document.createElement('div');
    messageDiv.className = `llm-chat-message ${role}`;

    // 用户消息：将代码引用 JSON 转换为缩略形式显示
    let displayContent = content;
    if (role === 'user') {
        displayContent = formatCodeRefsForDisplay(content);
    }

    // 直接显示内容，使用 pre 标签保留格式
    messageDiv.innerHTML = `<pre style="margin: 0; white-space: pre-wrap; word-break: break-word; font-family: inherit;">${escapeHtmlForChat(displayContent)}</pre>`;
    messagesContainer.appendChild(messageDiv);

    // 滚动到底部
    messagesContainer.scrollTop = messagesContainer.scrollHeight;

    // 添加到历史记录（保存原始内容用于发送给 LLM）
    chatHistory.push({ role, content });
}

/**
 * 将代码引用 JSON 转换为缩略形式显示
 * 例如：{"main.tex": {"1": "...", "2": "..."}} => [main.tex L1-2]
 * @param {string} content - 原始消息内容
 * @returns {string} 格式化后的内容
 */
function formatCodeRefsForDisplay(content) {
    if (!content) return '';

    // 尝试匹配 JSON 代码引用格式
    // 匹配 {"文件名": {"行号": "内容", ...}} 格式的 JSON
    const jsonPattern = /\{[\s\n]*"([^"]+)"[\s\n]*:[\s\n]*\{[\s\n]*(?:"(\d+)"[\s\n]*:[\s\n]*"[^"]*"[\s\n]*,?[\s\n]*)+\}[\s\n]*\}/g;

    return content.replace(jsonPattern, (match) => {
        try {
            const parsed = JSON.parse(match);
            const fileName = Object.keys(parsed)[0];
            if (!fileName) return match;

            const fileData = parsed[fileName];

            // 检查是否是图像或 PDF
            if (fileData && fileData.type === 'image') {
                return `[📷 ${fileName}]`;
            }
            if (fileData && fileData.type === 'pdf') {
                return `[📄 ${fileName}]`;
            }

            // 文本文件：提取行号范围
            const lineNumbers = Object.keys(fileData)
                .filter(k => !isNaN(parseInt(k)))
                .map(n => parseInt(n))
                .sort((a, b) => a - b);

            if (lineNumbers.length === 0) {
                return `[${fileName}]`;
            } else if (lineNumbers.length === 1) {
                return `[${fileName} L${lineNumbers[0]}]`;
            } else {
                return `[${fileName} L${lineNumbers[0]}-${lineNumbers[lineNumbers.length - 1]}]`;
            }
        } catch (e) {
            // JSON 解析失败，保持原样
            return match;
        }
    });
}

/**
 * 显示正在输入指示器
 */
function showTypingIndicator() {
    const messagesContainer = document.getElementById('llm-chat-messages');
    if (!messagesContainer) return;

    // 移除已有的指示器
    const existing = messagesContainer.querySelector('.typing-indicator-wrapper');
    if (existing) existing.remove();

    const indicatorDiv = document.createElement('div');
    indicatorDiv.className = 'llm-chat-message assistant typing-indicator-wrapper';
    indicatorDiv.innerHTML = `
        <div class="typing-indicator">
            <span></span>
            <span></span>
            <span></span>
        </div>
    `;
    messagesContainer.appendChild(indicatorDiv);
    messagesContainer.scrollTop = messagesContainer.scrollHeight;
}

/**
 * 隐藏正在输入指示器
 */
function hideTypingIndicator() {
    const messagesContainer = document.getElementById('llm-chat-messages');
    if (!messagesContainer) return;

    const indicator = messagesContainer.querySelector('.typing-indicator-wrapper');
    if (indicator) indicator.remove();
}


/**
 * 转义 HTML 特殊字符
 */
function escapeHtmlForChat(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML.replace(/\n/g, '<br>');
}

// =================== 结构化响应解析 ===================

/**
 * 解析LLM响应，分离消息和编辑指令
 * @param {string} response - LLM的原始响应
 * @returns {object} { message: string, edits: array }
 */
function parseAmbientResponse(response) {
    if (!response) {
        return { message: '', edits: [] };
    }

    // 尝试解析为JSON
    try {
        // 先尝试直接解析
        const parsed = JSON.parse(response);

        // 检查是否是结构化格式（包含message或edits字段）
        if (parsed && (parsed.message !== undefined || parsed.edits !== undefined)) {
            console.log('[AmbientWriting] Parsed structured response:', parsed);
            return {
                message: parsed.message || '',
                edits: Array.isArray(parsed.edits) ? parsed.edits : []
            };
        }
    } catch (e) {
        // 不是有效的JSON，尝试提取JSON部分
        // 有时LLM可能会在JSON前后添加一些文字
        const jsonMatch = response.match(/\{[\s\S]*"edits"[\s\S]*\}/);
        if (jsonMatch) {
            try {
                const parsed = JSON.parse(jsonMatch[0]);
                if (parsed && (parsed.message !== undefined || parsed.edits !== undefined)) {
                    console.log('[AmbientWriting] Extracted and parsed JSON from response:', parsed);
                    return {
                        message: parsed.message || '',
                        edits: Array.isArray(parsed.edits) ? parsed.edits : []
                    };
                }
            } catch (e2) {
                // 提取的内容也不是有效JSON
            }
        }
    }

    // 纯文本消息
    return {
        message: response,
        edits: []
    };
}

/**
 * 获取编辑类型的中文标签
 */
function getEditTypeLabel(type) {
    const labels = {
        'delete': '删除',
        'insert': '插入',
        'modify': '修改'
    };
    return labels[type] || type;
}

/**
 * 显示编辑确认弹窗（发送到Overleaf页面）
 * @param {object} edit - 编辑指令
 * @returns {Promise<{confirmed: boolean, success: boolean, error?: string}>}
 */
async function showEditPopup(edit) {
    return new Promise((resolve) => {
        chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
            if (!tabs || tabs.length === 0) {
                resolve({ confirmed: false, success: false, error: '无法获取当前标签页' });
                return;
            }

            chrome.tabs.sendMessage(tabs[0].id, {
                type: 'SHOW_EDIT_POPUP',
                edit: edit
            }, (response) => {
                if (chrome.runtime.lastError) {
                    console.error('[AmbientWriting] Error showing edit popup:', chrome.runtime.lastError);
                    resolve({ confirmed: false, success: false, error: chrome.runtime.lastError.message });
                    return;
                }
                resolve(response || { confirmed: false, success: false });
            });
        });
    });
}

/**
 * 依次处理编辑指令
 * 优先执行修改操作，再执行插入/删除操作，避免行号偏移问题
 * @param {array} edits - 编辑指令数组
 */
async function processEdits(edits) {
    if (!edits || edits.length === 0) return;

    console.log('[AmbientWriting] Processing', edits.length, 'edit(s)');

    // 分离修改操作和插入/删除操作
    const modifyEdits = edits.filter(e => e.type === 'modify');
    const insertDeleteEdits = edits.filter(e => e.type === 'insert' || e.type === 'delete');

    // 合并：先修改，后插入/删除
    const sortedEdits = [...modifyEdits, ...insertDeleteEdits];

    if (modifyEdits.length > 0 && insertDeleteEdits.length > 0) {
        console.log('[AmbientWriting] Reordered edits: modify first, then insert/delete');
        addChatMessage(`📋 检测到混合操作，将先执行 ${modifyEdits.length} 个修改，再执行 ${insertDeleteEdits.length} 个插入/删除`, 'system');
    }

    for (let i = 0; i < sortedEdits.length; i++) {
        const edit = sortedEdits[i];
        const typeLabel = getEditTypeLabel(edit.type);

        console.log(`[AmbientWriting] Processing edit ${i + 1}/${sortedEdits.length}:`, edit);

        // 显示编辑弹窗并等待用户确认
        const result = await showEditPopup(edit);

        if (result.confirmed && result.success) {
            addChatMessage(`✓ 已应用${typeLabel}操作`, 'system');
        } else if (result.confirmed && !result.success) {
            addChatMessage(`✗ ${typeLabel}操作失败: ${result.error || '未知错误'}`, 'system');
        } else {
            addChatMessage(`⊘ 已跳过${typeLabel}操作`, 'system');
        }
    }
}

/**
 * 清空对话
 */
function clearChat() {
    const messagesContainer = document.getElementById('llm-chat-messages');
    if (!messagesContainer) return;

    messagesContainer.innerHTML = `
        <div class="llm-chat-message system">
            欢迎使用氛围写作助手！你可以：<br>
            • 直接输入问题进行对话<br>
            • 在 Overleaf 选中文本后点击"插入氛围写作"按钮引用代码<br>
            • 按 Enter 发送消息，Shift+Enter 换行
        </div>
    `;
    chatHistory = [];

    // 清空输入框
    const input = document.getElementById('llm-chat-input');
    if (input) {
        input.innerHTML = '';
    }
}

/**
 * 发送消息到 LLM
 */
async function sendMessage() {
    const input = document.getElementById('llm-chat-input');
    const sendBtn = document.getElementById('llm-send-btn');

    if (!input || !sendBtn) return;

    // 获取结构化内容（分离文本、图像和 PDF）
    const structured = getInputContentStructured();
    const userMessage = structured.text.trim();

    // 如果没有文本内容、图像和 PDF，不发送
    if (!userMessage && structured.images.length === 0 && structured.pdfs.length === 0) return;

    // 禁用发送按钮
    sendBtn.disabled = true;
    input.setAttribute('contenteditable', 'false');

    // 添加用户消息（显示原始输入内容）
    addChatMessage(userMessage, 'user');

    // 清空输入框
    input.innerHTML = '';

    // 显示正在输入
    showTypingIndicator();

    try {
        // 检查是否为绘图模式
        if (isDrawMode) {
            await sendDrawMessage(userMessage, structured);
        } else {
            await sendChatMessage(userMessage, structured);
        }
    } catch (error) {
        hideTypingIndicator();
        console.error('[AmbientWriting] Error sending message:', error);
        addChatMessage(`发送失败: ${error.message}${AI_MODEL_ERROR_HINT_AMBIENT}`, 'system');
    } finally {
        sendBtn.disabled = false;
        input.setAttribute('contenteditable', 'true');
        input.focus();
    }
}

/**
 * 发送普通聊天消息
 */
async function sendChatMessage(userMessage, structured) {
    // 获取氛围写作专用的模型和思考程度（而不是翻译模型）
    const model = await getSelectedAmbientModel();
    const reasoningEffort = await getSelectedAmbientReasoningEffort();

    // 构建请求 - 发送结构化数据
    const requestBody = {
        message: userMessage,
        history: chatHistory.slice(-10), // 只发送最近10条消息
        images: structured.images,  // 图像数组 [{fileName, base64, mimeType}]
        pdfs: structured.pdfs,      // PDF 数组 [{fileName, base64}]
        model: model,               // 模型选择
        reasoning_effort: reasoningEffort  // 思考程度
    };

    console.log('[AmbientWriting] Sending message with', structured.images.length, 'images,', structured.pdfs.length, 'pdfs, model:', model, ', reasoningEffort:', reasoningEffort);

    // 调用后端 API
    const response = await fetch(`${BACKEND_API_URL}/api/chat/ambient`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        credentials: 'include',
        body: JSON.stringify(requestBody)
    });

    hideTypingIndicator();

    if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.detail || `请求失败: ${response.status}`);
    }

    const result = await response.json();

    // 解析响应，分离消息和编辑指令
    const parsed = parseAmbientResponse(result.reply);

    // 显示对话消息（如果有）
    if (parsed.message) {
        addChatMessage(parsed.message, 'assistant');
    }

    // 如果有编辑指令，将完整响应记录到历史（用于上下文）
    // 这样LLM知道它之前做了什么修改
    if (parsed.edits && parsed.edits.length > 0) {
        // 如果已经记录了message，需要更新最后一条历史为完整响应
        if (parsed.message && chatHistory.length > 0) {
            // 替换最后一条assistant消息为完整的JSON响应
            chatHistory[chatHistory.length - 1].content = result.reply;
        } else if (!parsed.message) {
            // 如果没有message但有edits，直接记录完整响应
            chatHistory.push({ role: 'assistant', content: result.reply });
        }
    }

    // 更新剩余 tokens
    if (result.remaining_tokens !== undefined) {
        const tokenDisplay = document.getElementById('user-tokens');
        if (tokenDisplay) {
            tokenDisplay.textContent = result.remaining_tokens;
        }
    }

    // 处理编辑指令（如果有）
    if (parsed.edits && parsed.edits.length > 0) {
        await processEdits(parsed.edits);
    }
}

/**
 * 发送绘图消息
 */
async function sendDrawMessage(prompt, structured) {
    const imageCount = structured?.images?.length || 0;
    console.log('[AmbientWriting] Drawing mode - sending prompt:', prompt, 'with', imageCount, 'images');

    // 构建请求体
    const requestBody = { prompt };

    // 如果有图像输入，添加到请求中
    if (structured?.images && structured.images.length > 0) {
        requestBody.images = structured.images.map(img => ({
            base64: img.base64,
            mimeType: img.mimeType || 'image/png'
        }));
    }

    // 调用绘图 API
    const response = await fetch(`${BACKEND_API_URL}/api/draw/generate`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        credentials: 'include',
        body: JSON.stringify(requestBody)
    });

    hideTypingIndicator();

    if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.detail || `绘图请求失败: ${response.status}`);
    }

    const result = await response.json();

    if (!result.success) {
        throw new Error(result.error || '绘图失败');
    }

    // 更新剩余 tokens
    if (result.remaining_tokens !== undefined) {
        const tokenDisplay = document.getElementById('user-tokens');
        if (tokenDisplay) {
            tokenDisplay.textContent = result.remaining_tokens;
        }
    }

    // 显示生成的图片
    const imageDataUrl = `data:image/${result.image_format};base64,${result.image_base64}`;
    addImageMessage(imageDataUrl, result.image_format);

    // 生成文件名 (使用简单格式，避免 Overleaf 的文件名验证问题)
    const timestamp = Date.now();
    const fileName = `img_${timestamp}.${result.image_format}`;

    // 添加操作按钮消息
    addChatMessage(`图片已生成 (消耗 ${result.consumed_points} 点)\n点击下方按钮下载图片:`, 'system');
    addDownloadButton(result.image_base64, result.image_format, fileName);
}

/**
 * 添加图片消息到对话框
 */
function addImageMessage(imageDataUrl, format) {
    const messagesContainer = document.getElementById('llm-chat-messages');
    if (!messagesContainer) return;

    const messageDiv = document.createElement('div');
    messageDiv.className = 'llm-chat-message assistant';
    messageDiv.innerHTML = `
        <div style="text-align: center;">
            <img src="${imageDataUrl}" alt="Generated image" style="max-width: 100%; max-height: 300px; border-radius: 8px; cursor: pointer;" onclick="window.open('${imageDataUrl}', '_blank')">
            <p style="margin-top: 8px; font-size: 12px; color: #64748b;">点击图片查看大图</p>
        </div>
    `;
    messagesContainer.appendChild(messageDiv);
    messagesContainer.scrollTop = messagesContainer.scrollHeight;
}

/**
 * 添加下载按钮到对话框
 */
function addDownloadButton(base64Data, format, fileName) {
    const messagesContainer = document.getElementById('llm-chat-messages');
    if (!messagesContainer) return;

    const buttonDiv = document.createElement('div');
    buttonDiv.className = 'llm-chat-message system';
    buttonDiv.style.textAlign = 'center';
    buttonDiv.innerHTML = `
        <button class="download-image-btn" style="
            padding: 8px 16px;
            background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 13px;
            font-weight: 500;
            cursor: pointer;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            transition: all 0.2s;
        ">
            <svg width="16" height="16" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4"></path>
            </svg>
            下载图片
        </button>
    `;

    const button = buttonDiv.querySelector('button');
    button.addEventListener('click', () => {
        try {
            // 创建下载链接
            const link = document.createElement('a');
            link.href = `data:image/${format};base64,${base64Data}`;
            link.download = fileName;
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);

            // 更新按钮状态
            button.style.background = '#16a34a';
            button.innerHTML = `
                <svg width="16" height="16" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                </svg>
                下载完成！
            `;

            // 3秒后恢复按钮
            setTimeout(() => {
                button.style.background = 'linear-gradient(135deg, #3b82f6 0%, #2563eb 100%)';
                button.innerHTML = `
                    <svg width="16" height="16" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4"></path>
                    </svg>
                    下载图片
                `;
            }, 3000);
        } catch (error) {
            console.error('[AmbientWriting] Download failed:', error);
            button.style.background = '#ef4444';
            button.innerHTML = `下载失败: ${error.message}`;
        }
    });

    messagesContainer.appendChild(buttonDiv);
    messagesContainer.scrollTop = messagesContainer.scrollHeight;
}

// =================== 代码引用功能 ===================

/**
 * 创建代码标签 HTML
 * @param {object} data - 代码数据 {"文件名": {"行号": "内容", ...}} 或 {"文件名": {type: "image"|"pdf", base64: "..."}}
 * @returns {string} HTML 字符串
 */
function createCodeChipHTML(data) {
    const fileName = Object.keys(data)[0] || 'unknown';
    const fileData = data[fileName] || {};

    let labelText = fileName;
    let iconSvg;

    // 检查文件类型
    if (fileData.type === 'image') {
        // 图像文件
        labelText = `📷 ${fileName}`;
        iconSvg = `<svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"></path></svg>`;
    } else if (fileData.type === 'pdf') {
        // PDF 文件
        labelText = `📄 ${fileName}`;
        iconSvg = `<svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 21h10a2 2 0 002-2V9.414a1 1 0 00-.293-.707l-5.414-5.414A1 1 0 0012.586 3H7a2 2 0 00-2 2v14a2 2 0 002 2z"></path></svg>`;
    } else {
        // 文本文件：显示行号范围
        const lineNumbers = Object.keys(fileData).filter(k => !isNaN(parseInt(k))).map(n => parseInt(n)).sort((a, b) => a - b);
        if (lineNumbers.length > 0) {
            if (lineNumbers.length === 1) {
                labelText = `${fileName} L${lineNumbers[0]}`;
            } else {
                labelText = `${fileName} L${lineNumbers[0]}-${lineNumbers[lineNumbers.length - 1]}`;
            }
        }
        iconSvg = `<svg fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 20l4-16m4 4l4 4-4 4M6 16l-4-4 4-4"></path></svg>`;
    }

    // 将数据存储在 data 属性中
    const jsonData = JSON.stringify(data).replace(/"/g, '&quot;');

    return `<span class="llm-code-chip" contenteditable="false" data-code="${jsonData}">${iconSvg}<span class="llm-code-chip-text">${labelText}</span><button class="llm-code-chip-remove" type="button">×</button></span>`;
}

/**
 * 在输入框末尾插入代码标签
 * @param {object} data - 代码数据
 */
function insertCodeChip(data) {
    const input = document.getElementById('llm-chat-input');
    if (!input) return;

    const chipHTML = createCodeChipHTML(data);

    // 创建临时元素来解析 HTML
    const temp = document.createElement('div');
    temp.innerHTML = chipHTML;
    const chip = temp.firstChild;

    // 追加到输入框末尾
    input.appendChild(chip);

    // 将光标移动到标签后面
    const selection = window.getSelection();
    const range = document.createRange();
    range.setStartAfter(chip);
    range.setEndAfter(chip);
    selection.removeAllRanges();
    selection.addRange(range);

    input.focus();
}

/**
 * 在光标位置插入纯文本
 * @param {string} text - 要插入的文本
 */
function insertTextAtCursor(text) {
    const selection = window.getSelection();
    if (selection.rangeCount > 0) {
        const range = selection.getRangeAt(0);
        range.deleteContents();
        range.insertNode(document.createTextNode(text));
        range.collapse(false);
    }
}

/**
 * 获取输入框内容，返回结构化数据
 * @returns {object} { text: string, images: [{fileName, base64, mimeType}], pdfs: [{fileName, base64}], textRefs: [{fileName, lines}] }
 */
function getInputContentStructured() {
    const input = document.getElementById('llm-chat-input');
    if (!input) return { text: '', images: [], pdfs: [], textRefs: [] };

    const images = [];
    const pdfs = [];
    const textRefs = [];

    // 克隆节点以避免修改原始内容
    const clone = input.cloneNode(true);

    // 处理代码标签，分离图像、PDF 和文本引用
    const chips = clone.querySelectorAll('.llm-code-chip');
    chips.forEach(chip => {
        const jsonData = chip.getAttribute('data-code');
        if (jsonData) {
            try {
                const data = JSON.parse(jsonData.replace(/&quot;/g, '"'));
                const fileName = Object.keys(data)[0];
                const fileData = data[fileName];

                if (fileData && fileData.type === 'image') {
                    // 图像文件
                    images.push({
                        fileName: fileName,
                        base64: fileData.base64,
                        mimeType: fileData.mimeType || 'image/png'
                    });
                    chip.replaceWith(`[图像: ${fileName}]`);
                } else if (fileData && fileData.type === 'pdf') {
                    // PDF 文件
                    pdfs.push({
                        fileName: fileName,
                        base64: fileData.base64
                    });
                    chip.replaceWith(`[PDF: ${fileName}]`);
                } else {
                    // 文本文件引用
                    textRefs.push({
                        fileName: fileName,
                        lines: fileData
                    });
                    const jsonText = JSON.stringify(data, null, 2);
                    chip.replaceWith(jsonText);
                }
            } catch (e) {
                chip.replaceWith('[引用]');
            }
        }
    });

    // 获取纯文本内容
    const text = clone.textContent || clone.innerText || '';

    return { text, images, pdfs, textRefs };
}

/**
 * 获取输入框内容（兼容旧版，返回纯文本）
 */
function getInputContent() {
    const structured = getInputContentStructured();
    return structured.text;
}

// =================== 事件绑定 ===================

/**
 * 初始化氛围写作功能
 */
function initAmbientWriting() {
    console.log('[AmbientWriting] Initializing...');

    // 氛围写作按钮
    const ambientBtn = document.getElementById('ambient-writing-btn');
    if (ambientBtn) {
        ambientBtn.addEventListener('click', toggleAmbientWritingMode);
    }

    // 退出按钮
    const exitBtn = document.getElementById('llm-exit-btn');
    if (exitBtn) {
        exitBtn.addEventListener('click', exitAmbientWritingMode);
    }

    // 清空按钮
    const clearBtn = document.getElementById('llm-clear-chat-btn');
    if (clearBtn) {
        clearBtn.addEventListener('click', clearChat);
    }

    // 绘图模式开关
    const drawToggle = document.getElementById('draw-mode-toggle');
    if (drawToggle) {
        drawToggle.addEventListener('change', async (e) => {
            // 如果要开启绘图模式，先检查 VIP 等级
            if (e.target.checked) {
                try {
                    const vipStatus = await authService.checkVipStatus();
                    if (vipStatus.vipLevel !== 3) {
                        // 不是 Pro Max 会员，阻止开启绘图模式
                        e.target.checked = false;
                        addChatMessage('⚠️ 绘图功能仅限 Pro Max 会员使用，请升级会员后再试', 'system');
                        return;
                    }
                } catch (error) {
                    console.error('[AmbientWriting] Failed to check VIP status:', error);
                    e.target.checked = false;
                    addChatMessage('⚠️ 无法验证会员状态，请确保已登录', 'system');
                    return;
                }
            }

            isDrawMode = e.target.checked;
            const toggleLabel = drawToggle.closest('.draw-mode-toggle');
            if (toggleLabel) {
                if (isDrawMode) {
                    toggleLabel.classList.add('active');
                } else {
                    toggleLabel.classList.remove('active');
                }
            }
            console.log('[AmbientWriting] Draw mode:', isDrawMode ? 'ON' : 'OFF');

            // 更新输入框提示
            const input = document.getElementById('llm-chat-input');
            if (input) {
                if (isDrawMode) {
                    input.setAttribute('data-placeholder', '基于Nano Banana Pro，会消耗大量点数，请谨慎使用。描述你想生成的图片...');
                } else {
                    input.setAttribute('data-placeholder', '在 Overleaf 选中文本后点击"插入氛围写作"，然后输入问题');
                }
            }
        });
    }

    // 发送按钮
    const sendBtn = document.getElementById('llm-send-btn');
    if (sendBtn) {
        sendBtn.addEventListener('click', sendMessage);
    }

    // 输入框回车发送
    const input = document.getElementById('llm-chat-input');
    if (input) {
        input.addEventListener('keydown', (e) => {
            if (e.key === 'Enter') {
                // Shift+Enter 换行
                if (e.shiftKey) {
                    // 允许默认行为（换行）
                    return;
                }
                // 普通 Enter 或 Ctrl+Enter 发送消息
                e.preventDefault();
                sendMessage();
            }
        });

        // 粘贴时只保留纯文本
        input.addEventListener('paste', (e) => {
            e.preventDefault();
            const text = e.clipboardData.getData('text/plain');
            document.execCommand('insertText', false, text);
        });

        // 事件委托：处理代码标签删除按钮点击
        input.addEventListener('click', (e) => {
            if (e.target.classList.contains('llm-code-chip-remove')) {
                e.preventDefault();
                e.stopPropagation();
                const chip = e.target.closest('.llm-code-chip');
                if (chip) {
                    chip.remove();
                }
            }
        });
    }

    // 监听来自 content.js 的消息
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
        if (request.type === 'INSERT_TO_AMBIENT_WRITING') {
            console.log('[AmbientWriting] Received insert request:', request);
            if (request.data) {
                // 自动进入氛围写作模式（如果尚未进入）
                if (!isAmbientWritingMode) {
                    enterAmbientWritingMode();
                }

                // 在输入框末尾插入代码标签
                setTimeout(() => {
                    insertCodeChip(request.data);
                    // 聚焦输入框
                    const input = document.getElementById('llm-chat-input');
                    if (input) input.focus();
                }, 100);
            }
        }

        // 处理文件树右键菜单的"插入到氛围写作"
        if (request.type === 'INSERT_FILE_TO_AMBIENT_WRITING') {
            console.log('[AmbientWriting] Received file insert request:', request);
            if (request.data) {
                // 自动进入氛围写作模式（如果尚未进入）
                if (!isAmbientWritingMode) {
                    enterAmbientWritingMode();
                }

                const fileName = request.data.fileName || 'unknown';
                const content = request.data.content || '';
                const contentType = request.data.contentType || 'text';

                let chipData = {};

                if (contentType === 'image') {
                    // 图片文件：存储 base64 数据
                    chipData[fileName] = {
                        type: 'image',
                        mimeType: request.data.mimeType || 'image/png',
                        base64: content  // data:image/xxx;base64,... 格式
                    };
                } else if (contentType === 'pdf') {
                    // PDF 文件：存储 base64 数据，后端会转换为图像
                    chipData[fileName] = {
                        type: 'pdf',
                        base64: content  // data:application/pdf;base64,... 格式
                    };
                } else {
                    // 文本文件：按行分割
                    const lines = content.split('\n');
                    const linesData = {};
                    lines.forEach((line, index) => {
                        linesData[index + 1] = line;
                    });
                    chipData[fileName] = linesData;
                }

                // 在输入框末尾插入代码标签
                setTimeout(() => {
                    insertCodeChip(chipData);
                    // 聚焦输入框
                    const input = document.getElementById('llm-chat-input');
                    if (input) input.focus();
                }, 100);
            }
        }

        // 处理语法检查请求（后台处理，不唤醒氛围写作界面）
        if (request.type === 'GRAMMAR_CHECK') {
            console.log('[AmbientWriting] Received grammar check request:', request.data);
            if (request.data) {
                // 构建代码引用 JSON
                const codeRefJson = JSON.stringify(request.data, null, 2);

                // 语法检查指令
                const grammarPrompt = `

请检查上述内容的语法错误，包括拼写错误、语法错误、标点符号错误等。
注意：只关注明确的错误，不需要提供润色或改进建议。如果发现错误，请直接给出修正。`;

                // 完整消息
                const userMessage = codeRefJson + grammarPrompt;

                // 后台发送消息，不切换到氛围写作界面
                sendBackgroundRequest(userMessage, '语法检查');
            }
        }

        // 处理一键润色请求（后台处理，不唤醒氛围写作界面）
        if (request.type === 'POLISH') {
            console.log('[AmbientWriting] Received polish request:', request.data);
            if (request.data) {
                // 构建代码引用 JSON
                const codeRefJson = JSON.stringify(request.data, null, 2);

                // 一键润色指令
                const polishPrompt = `

请对上述论文内容进行润色，使表达更加自然流畅，逻辑更加清晰通顺，更像人类撰写的高水平学术论文。
保持原文的核心意思不变，但可以调整句式结构、用词选择，使文章更加专业和地道。`;

                // 完整消息
                const userMessage = codeRefJson + polishPrompt;

                // 后台发送消息，不切换到氛围写作界面
                sendBackgroundRequest(userMessage, '一键润色');
            }
        }
    });

    console.log('[AmbientWriting] Initialized');
}

/**
 * 后台发送请求（不切换到氛围写作界面）
 * 用于语法检查和一键润色功能
 * @param {string} content - 要发送的消息内容
 * @param {string} taskName - 任务名称（用于显示提示）
 */
async function sendBackgroundRequest(content, taskName) {
    console.log('[AmbientWriting] Sending background request:', taskName);

    // 显示加载提示（在 Overleaf 页面）
    showLoadingIndicator(taskName);

    try {
        // 获取氛围写作专用的模型和思考程度（而不是翻译模型）
        const model = await getSelectedAmbientModel();
        const reasoningEffort = await getSelectedAmbientReasoningEffort();

        // 调用后端 API
        const response = await fetch(`${BACKEND_API_URL}/api/chat/ambient`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            credentials: 'include',
            body: JSON.stringify({
                message: content,
                history: [],  // 后台请求不使用历史记录
                images: [],
                model: model,
                reasoning_effort: reasoningEffort
            })
        });

        // 隐藏加载提示
        hideLoadingIndicator();

        if (!response.ok) {
            const errorData = await response.json().catch(() => ({}));
            throw new Error(errorData.detail || `请求失败: ${response.status}`);
        }

        const result = await response.json();

        // 解析响应
        const parsed = parseAmbientResponse(result.reply);

        // 更新剩余 tokens
        if (result.remaining_tokens !== undefined) {
            const tokenDisplay = document.getElementById('user-tokens');
            if (tokenDisplay) {
                tokenDisplay.textContent = result.remaining_tokens;
            }
        }

        // 处理编辑指令（如果有）
        if (parsed.edits && parsed.edits.length > 0) {
            await processEdits(parsed.edits);
        } else if (parsed.message) {
            // 如果没有编辑指令但有消息，显示结果弹窗
            showResultPopup(taskName, parsed.message);
        }

    } catch (error) {
        hideLoadingIndicator();
        console.error('[AmbientWriting] Background request error:', error);
        showResultPopup(taskName + ' 失败', error.message);
    }
}

/**
 * 在 Overleaf 页面显示加载提示
 * @param {string} taskName - 任务名称
 */
function showLoadingIndicator(taskName) {
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (!tabs || tabs.length === 0) return;
        chrome.tabs.sendMessage(tabs[0].id, {
            type: 'SHOW_LOADING_INDICATOR',
            taskName: taskName
        }).catch(() => {});
    });
}

/**
 * 隐藏 Overleaf 页面的加载提示
 */
function hideLoadingIndicator() {
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (!tabs || tabs.length === 0) return;
        chrome.tabs.sendMessage(tabs[0].id, {
            type: 'HIDE_LOADING_INDICATOR'
        }).catch(() => {});
    });
}

/**
 * 在 Overleaf 页面显示结果弹窗
 * @param {string} title - 标题
 * @param {string} message - 消息内容
 */
function showResultPopup(title, message) {
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (!tabs || tabs.length === 0) return;
        chrome.tabs.sendMessage(tabs[0].id, {
            type: 'SHOW_RESULT_POPUP',
            title: title,
            message: message
        }).catch(() => {});
    });
}

/**
 * 直接发送指定内容的消息（用于氛围写作界面内的请求）
 * @param {string} content - 要发送的消息内容
 */
async function sendMessageWithContent(content) {
    const sendBtn = document.getElementById('llm-send-btn');
    if (!sendBtn) return;

    // 禁用发送按钮
    sendBtn.disabled = true;

    // 添加用户消息到对话框（只显示简短提示）
    const displayMessage = content.length > 200 ? content.substring(0, 200) + '...' : content;
    addChatMessage(displayMessage, 'user');

    // 显示正在输入
    showTypingIndicator();

    try {
        // 获取氛围写作专用的模型和思考程度（而不是翻译模型）
        const model = await getSelectedAmbientModel();
        const reasoningEffort = await getSelectedAmbientReasoningEffort();

        // 调用后端 API
        const response = await fetch(`${BACKEND_API_URL}/api/chat/ambient`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            credentials: 'include',
            body: JSON.stringify({
                message: content,
                history: chatHistory.slice(-10),
                images: [],
                model: model,
                reasoning_effort: reasoningEffort
            })
        });

        hideTypingIndicator();

        if (!response.ok) {
            const errorData = await response.json().catch(() => ({}));
            throw new Error(errorData.detail || `请求失败: ${response.status}`);
        }

        const result = await response.json();

        // 解析响应
        const parsed = parseAmbientResponse(result.reply);

        // 显示对话消息（如果有）
        if (parsed.message) {
            addChatMessage(parsed.message, 'assistant');
        }

        // 更新剩余 tokens
        if (result.remaining_tokens !== undefined) {
            const tokenDisplay = document.getElementById('user-tokens');
            if (tokenDisplay) {
                tokenDisplay.textContent = result.remaining_tokens;
            }
        }

        // 处理编辑指令（如果有）
        if (parsed.edits && parsed.edits.length > 0) {
            await processEdits(parsed.edits);
        }

    } catch (error) {
        hideTypingIndicator();
        console.error('[AmbientWriting] Error sending message:', error);
        addChatMessage(`发送失败: ${error.message}${AI_MODEL_ERROR_HINT_AMBIENT}`, 'system');
    } finally {
        sendBtn.disabled = false;
    }
}

// 在 DOM 加载完成后初始化
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initAmbientWriting);
} else {
    initAmbientWriting();
}
