/**
 * 认证服务模块
 * 处理用户登录、登出、令牌管理等功能
 */

const AUTH_CONFIG = {
    // API基础URL - 使用Homepage项目的服务器
    API_BASE_URL: 'https://overleaf.top',

    // 本地开发环境
    // API_BASE_URL: 'http://localhost:8080',

    // 微信OAuth配置
    WECHAT_APP_ID: 'wxb237a52944eee3b4',

    // 存储键名
    STORAGE_KEYS: {
        ACCESS_TOKEN: 'access_token',
        USER_INFO: 'user_info',
        LOGIN_STATE: 'login_state'
    }
};

class AuthService {
    constructor() {
        this.token = null;
        this.userInfo = null;
        this.isInitialized = false;
    }

    /**
     * 初始化认证服务
     */
    async init() {
        if (this.isInitialized) return;

        try {
            // 从存储中恢复会话
            const stored = await chrome.storage.local.get([
                AUTH_CONFIG.STORAGE_KEYS.ACCESS_TOKEN,
                AUTH_CONFIG.STORAGE_KEYS.USER_INFO
            ]);

            if (stored[AUTH_CONFIG.STORAGE_KEYS.ACCESS_TOKEN]) {
                this.token = stored[AUTH_CONFIG.STORAGE_KEYS.ACCESS_TOKEN];
                this.userInfo = stored[AUTH_CONFIG.STORAGE_KEYS.USER_INFO];

                // 验证令牌是否仍然有效
                await this.validateToken();
            }

            this.isInitialized = true;
        } catch (error) {
            console.error('Failed to initialize auth service:', error);
            this.isInitialized = true;
        }
    }

    /**
     * 开始微信登录流程
     */
    async startWechatLogin() {
        try {
            // 直接打开 overleaf.top 的首页，让用户在那里登录
            const loginUrl = `${AUTH_CONFIG.API_BASE_URL}`;

            // 在新标签页中打开页面
            await chrome.tabs.create({
                url: loginUrl,
                active: true
            });

            // 不需要等待，让用户自己在 overleaf.top 上完成登录
            // 登录成功后，用户可以刷新插件来获取登录状态
            return { message: '请在 overleaf.top 网站上完成登录' };
        } catch (error) {
            console.error('Failed to open login page:', error);
            throw error;
        }
    }

    /**
     * 从 overleaf.top 同步登录状态
     */
    async syncLoginStatus() {
        try {
            // 尝试从 overleaf.top 获取用户信息
            const response = await fetch(`${AUTH_CONFIG.API_BASE_URL}/api/user/me`, {
                method: 'GET',
                credentials: 'include',
                headers: {
                    'Content-Type': 'application/json'
                }
            });

            if (response.ok) {
                const userInfo = await response.json();

                // 创建一个简单的token（实际应该从cookie或响应头获取）
                const sessionData = {
                    token: 'cookie-session', // 使用cookie认证
                    user: userInfo
                };

                await this.saveUserSession(sessionData);
                return userInfo;
            } else {
                // 未登录
                await this.logout();
                return null;
            }
        } catch (error) {
            console.error('Failed to sync login status:', error);
            return null;
        }
    }

    /**
     * 从回调页面提取用户信息
     */
    async extractUserInfoFromCallbackPage(tabId) {
        try {
            // 注入脚本获取页面中的用户信息
            const [result] = await chrome.scripting.executeScript({
                target: { tabId: tabId },
                func: () => {
                    // 尝试从页面中获取用户信息
                    // Homepage项目会在回调页面设置这些信息
                    const userInfoElement = document.getElementById('user-info');
                    if (userInfoElement) {
                        return JSON.parse(userInfoElement.textContent);
                    }

                    // 备选：从cookie中获取
                    const getCookie = (name) => {
                        const value = `; ${document.cookie}`;
                        const parts = value.split(`; ${name}=`);
                        if (parts.length === 2) {
                            return parts.pop().split(';').shift();
                        }
                        return null;
                    };

                    const token = getCookie('access_token');
                    if (token) {
                        return { token };
                    }

                    return null;
                }
            });

            if (result && result.result) {
                // 如果只有token，获取完整用户信息
                if (result.result.token && !result.result.user) {
                    const userInfo = await this.fetchUserInfo(result.result.token);
                    return { token: result.result.token, user: userInfo };
                }
                return result.result;
            }

            return null;
        } catch (error) {
            console.error('Failed to extract user info:', error);
            return null;
        }
    }

    /**
     * 获取当前用户信息
     */
    async getCurrentUser() {
        if (!this.isInitialized) {
            await this.init();
        }

        if (!this.token) {
            return null;
        }

        try {
            // 如果已有缓存的用户信息，直接返回
            if (this.userInfo) {
                return this.userInfo;
            }

            // 从API获取最新用户信息
            const userInfo = await this.fetchUserInfo(this.token);
            this.userInfo = userInfo;

            // 更新存储
            await chrome.storage.local.set({
                [AUTH_CONFIG.STORAGE_KEYS.USER_INFO]: userInfo
            });

            return userInfo;
        } catch (error) {
            console.error('Failed to get current user:', error);
            // 令牌可能已失效
            await this.logout();
            return null;
        }
    }

    /**
     * 从API获取用户信息
     */
    async fetchUserInfo(token) {
        const response = await fetch(`${AUTH_CONFIG.API_BASE_URL}/api/user/me`, {
            method: 'GET',
            headers: {
                'Authorization': `Bearer ${token}`,
                'Content-Type': 'application/json'
            },
            credentials: 'include'
        });

        if (!response.ok) {
            throw new Error('Failed to fetch user info');
        }

        return await response.json();
    }

    /**
     * 验证令牌是否有效
     */
    async validateToken() {
        if (!this.token) {
            return false;
        }

        try {
            const userInfo = await this.fetchUserInfo(this.token);
            this.userInfo = userInfo;

            // 更新存储中的用户信息
            await chrome.storage.local.set({
                [AUTH_CONFIG.STORAGE_KEYS.USER_INFO]: userInfo
            });

            return true;
        } catch (error) {
            console.error('Token validation failed:', error);
            // 令牌无效，清除会话
            await this.logout();
            return false;
        }
    }

    /**
     * 保存用户会话
     */
    async saveUserSession(sessionData) {
        this.token = sessionData.token;
        this.userInfo = sessionData.user;

        // 保存到chrome.storage
        await chrome.storage.local.set({
            [AUTH_CONFIG.STORAGE_KEYS.ACCESS_TOKEN]: sessionData.token,
            [AUTH_CONFIG.STORAGE_KEYS.USER_INFO]: sessionData.user,
            [AUTH_CONFIG.STORAGE_KEYS.LOGIN_STATE]: true
        });

        // 通知其他组件登录状态变化
        await this.notifyAuthStateChange(true);
    }

    /**
     * 登出
     */
    async logout() {
        try {
            // 调用API登出
            if (this.token) {
                await fetch(`${AUTH_CONFIG.API_BASE_URL}/api/user/logout`, {
                    method: 'POST',
                    headers: {
                        'Authorization': `Bearer ${this.token}`
                    },
                    credentials: 'include'
                });
            }
        } catch (error) {
            console.error('API logout failed:', error);
        }

        // 清除本地会话
        this.token = null;
        this.userInfo = null;

        // 清除存储
        await chrome.storage.local.remove([
            AUTH_CONFIG.STORAGE_KEYS.ACCESS_TOKEN,
            AUTH_CONFIG.STORAGE_KEYS.USER_INFO,
            AUTH_CONFIG.STORAGE_KEYS.LOGIN_STATE
        ]);

        // 通知其他组件登录状态变化
        await this.notifyAuthStateChange(false);
    }

    /**
     * 通知认证状态变化
     */
    async notifyAuthStateChange(isLoggedIn) {
        // 向所有标签页发送消息
        const tabs = await chrome.tabs.query({});
        for (const tab of tabs) {
            try {
                chrome.tabs.sendMessage(tab.id, {
                    type: 'AUTH_STATE_CHANGED',
                    isLoggedIn: isLoggedIn,
                    userInfo: isLoggedIn ? this.userInfo : null
                });
            } catch (error) {
                // 忽略无法发送消息的标签页
            }
        }

        // 向扩展内部发送消息
        chrome.runtime.sendMessage({
            type: 'AUTH_STATE_CHANGED',
            isLoggedIn: isLoggedIn,
            userInfo: isLoggedIn ? this.userInfo : null
        });
    }

    /**
     * 检查是否已登录
     */
    async isLoggedIn() {
        if (!this.isInitialized) {
            await this.init();
        }
        return !!this.token;
    }

    /**
     * 获取访问令牌
     */
    getAccessToken() {
        return this.token;
    }

    /**
     * 为API请求添加认证头
     */
    getAuthHeaders() {
        if (!this.token) {
            return {};
        }

        return {
            'Authorization': `Bearer ${this.token}`
        };
    }

    /**
     * 检查VIP状态
     */
    async checkVipStatus() {
        const user = await this.getCurrentUser();
        if (!user) {
            return {
                isVip: false,
                vipLevel: 0,
                expireAt: null,
                remainingTokens: 0
            };
        }

        const now = new Date();
        const expireAt = user.vip_expire_at ? new Date(user.vip_expire_at) : null;
        const isVipActive = expireAt && expireAt > now;

        let vipLevel = 0;
        if (isVipActive) {
            if (user.is_vip3) vipLevel = 3;
            else if (user.is_vip2) vipLevel = 2;
            else if (user.is_vip1) vipLevel = 1;
        }

        return {
            isVip: isVipActive,
            vipLevel: vipLevel,
            expireAt: expireAt,
            remainingTokens: user.remaining_tokens || 0,
            vipType: vipLevel === 3 ? 'Pro Max' : vipLevel === 2 ? 'Pro' : vipLevel === 1 ? '普通会员' : '免费用户'
        };
    }

    /**
     * 格式化日期显示
     */
    formatDate(date) {
        if (!date) return '';
        const d = new Date(date);
        return d.toLocaleDateString('zh-CN', {
            year: 'numeric',
            month: '2-digit',
            day: '2-digit'
        });
    }

    /**
     * 计算剩余天数
     */
    calculateRemainingDays(expireDate) {
        if (!expireDate) return 0;
        const now = new Date();
        const expire = new Date(expireDate);
        const diff = expire - now;
        return Math.max(0, Math.ceil(diff / (1000 * 60 * 60 * 24)));
    }
}

// 创建全局实例
const authService = new AuthService();

// 导出给其他模块使用
if (typeof module !== 'undefined' && module.exports) {
    module.exports = authService;
}