// Content script for Overleaf Document Reader Extension
// This script runs in the context of Overleaf project pages

(function() {
    'use strict';

    let documentContent = '';
    let currentFileName = '';
    let selectedText = '';
    let selectedRange = null;

    // Compact view state
    let compactViewPanel = null;
    let compactViewContent = '';
    let compactPanelHeight = '33.33vh';
    let isResizing = false;

    // Function to inject script that will access the editor
    function injectEditorAccessScript() {
        // Use external script file to avoid CSP issues
        const script = document.createElement('script');
        script.src = chrome.runtime.getURL('core/injected.js');
        script.onload = function() {
            console.log('[Overleaf Reader] External script loaded successfully');
            this.remove();
        };
        script.onerror = function() {
            console.error('[Overleaf Reader] Failed to load external script');
            // Fallback to DOM inspection
            tryDOMInspection();
        };
        (document.head || document.documentElement).appendChild(script);
    }

    // DOM inspection disabled - only use injected script for reliable content
    function tryDOMInspection() {
        // Do nothing - content is obtained from injected script only
        console.log('[Overleaf Reader] DOM inspection skipped, waiting for injected script');
    }

    // Function to jump to a specific line in the editor
    function jumpToLine(lineNumber) {
        console.log('[Overleaf Reader] Attempting to jump to line:', lineNumber);

        // Method 1: Try injecting script to access CodeMirror (most reliable)
        window.postMessage({
            type: 'OVERLEAF_READER_JUMP_TO_LINE',
            lineNumber: lineNumber
        }, '*');

        // Method 2: Try to use CodeMirror 6 API directly
        const cmElements = document.querySelectorAll('.cm-editor');
        for (const cmElement of cmElements) {
            if (cmElement.cmView) {
                try {
                    const view = cmElement.cmView;
                    const doc = view.state.doc;

                    if (lineNumber > doc.lines) {
                        console.log('[Overleaf Reader] Line number out of range:', lineNumber, 'max:', doc.lines);
                        continue;
                    }

                    const line = doc.line(lineNumber);
                    const pos = line.from;

                    view.dispatch({
                        selection: { anchor: pos, head: pos },
                        scrollIntoView: true
                    });

                    view.focus();
                    console.log('[Overleaf Reader] Successfully jumped to line:', lineNumber);
                    return true;
                } catch (e) {
                    console.log('[Overleaf Reader] Error with direct API:', e);
                }
            }
        }

        console.log('[Overleaf Reader] Jump attempt completed');
        return true; // Return true as we tried our best
    }

    // Function to update editor content
    function updateEditorContent(newContent, originalContent) {
        console.log('[Overleaf Reader] Attempting to update content, length:', newContent.length);

        // Method 1: Try injecting script to access CodeMirror (most reliable)
        window.postMessage({
            type: 'OVERLEAF_READER_UPDATE_CONTENT',
            content: newContent,
            originalContent: originalContent
        }, '*');

        // Method 2: Try to use CodeMirror 6 API directly
        const cmElements = document.querySelectorAll('.cm-editor');
        for (const cmElement of cmElements) {
            if (cmElement.cmView) {
                try {
                    const view = cmElement.cmView;
                    const doc = view.state.doc;

                    // Replace entire document content
                    view.dispatch({
                        changes: {
                            from: 0,
                            to: doc.length,
                            insert: newContent
                        }
                    });

                    view.focus();
                    console.log('[Overleaf Reader] Successfully updated content via direct API');
                    return true;
                } catch (e) {
                    console.log('[Overleaf Reader] Error with direct API update:', e);
                }
            }
        }

        console.log('[Overleaf Reader] Update attempt completed');
        return true; // Return true as we tried our best
    }

    // Function to update document content
    function updateContent(content, source) {
        if (content && content !== documentContent) {
            console.log('[Overleaf Reader] Content updated from', source, 'length:', content.length);
            documentContent = content;

            // Send to background script
            chrome.runtime.sendMessage({
                type: 'DOCUMENT_CONTENT_UPDATED',
                content: documentContent,
                timestamp: Date.now()
            }).catch(err => {
                console.error('[Overleaf Reader] Error sending to background:', err);
            });
        }
    }

    // Text selection tracking
    document.addEventListener('mouseup', () => {
        setTimeout(() => {
            const selection = window.getSelection();
            const newSelectedText = selection.toString().trim();

            if (newSelectedText !== selectedText) {
                selectedText = newSelectedText;
                selectedRange = newSelectedText ? selection.getRangeAt(0).cloneRange() : null;

                // Send selection update to sidepanel
                chrome.runtime.sendMessage({
                    type: 'SELECTION_UPDATE',
                    selectedText: selectedText,
                    source: 'editor'
                }).catch(() => {
                    // Sidepanel might not be open, ignore error
                });
            }
        }, 10);
    });

    // Listen for messages from injected script and iframe
    window.addEventListener('message', function(event) {
        // Handle message from compact view iframe to return to sidepanel
        if (event.data && event.data.type === 'COMPACT_VIEW_RETURN_TO_SIDEPANEL') {
            console.log('[Content] Received request to return to sidepanel');
            // Close compact view
            closeCompactView();
            // Open sidepanel
            chrome.runtime.sendMessage({
                type: 'OPEN_SIDEPANEL_FROM_BUTTON'
            });
            return;
        }

        if (event.data && event.data.type === 'OVERLEAF_READER_CONTENT') {
            updateContent(event.data.content, event.data.source || 'injected');

            // Request current filename
            console.log('[Overleaf Reader] Requesting filename from injected script...');
            window.postMessage({ type: 'OVERLEAF_READER_GET_FILENAME' }, '*');
        } else if (event.data && event.data.type === 'OVERLEAF_READER_FILENAME_RESPONSE') {
            console.log('[Overleaf Reader] Received filename response:', event.data.filename);
            if (event.data.filename) {
                currentFileName = event.data.filename;
                console.log('[Overleaf Reader] ✓ Set currentFileName to:', currentFileName);

                // Notify sidepanel about filename update
                chrome.runtime.sendMessage({
                    type: 'FILENAME_UPDATED',
                    filename: currentFileName
                }).catch(() => {
                    console.log('[Overleaf Reader] Sidepanel might not be open');
                });
            } else {
                console.log('[Overleaf Reader] ✗ Filename response was empty');
            }
        } else if (event.data && event.data.type === 'OVERLEAF_CURSOR_LINE_CHANGED') {
            // Forward cursor line change to sidepanel
            chrome.runtime.sendMessage({
                type: 'EDITOR_CURSOR_LINE_CHANGED',
                lineNumber: event.data.lineNumber
            }).catch(() => {
                // Sidepanel might not be open, ignore error
            });
        } else if (event.data && event.data.type === 'OVERLEAF_LINES_INSERTED') {
            // Forward line insertion to sidepanel
            console.log('[Content] Forwarding EDITOR_LINES_INSERTED:', event.data.lineNumber, event.data.count, 'type:', event.data.insertType, 'source:', event.data.source, 'file:', currentFileName);
            chrome.runtime.sendMessage({
                type: 'EDITOR_LINES_INSERTED',
                lineNumber: event.data.lineNumber,
                count: event.data.count,
                insertType: event.data.insertType || 'end',
                prevLineNumber: event.data.prevLineNumber,
                currentLineNumber: event.data.currentLineNumber,
                source: event.data.source,  // 传递来源标记
                fileName: currentFileName   // 附带当前文件名，用于防止文件切换时的竞态条件
            }).catch((e) => {
                console.log('[Content] Failed to send EDITOR_LINES_INSERTED:', e);
            });
        } else if (event.data && event.data.type === 'OVERLEAF_LINES_DELETED') {
            // Forward line deletion to sidepanel
            console.log('[Content] Forwarding EDITOR_LINES_DELETED:', event.data.lineNumber, event.data.count, 'type:', event.data.deleteType, 'source:', event.data.source, 'file:', currentFileName);
            chrome.runtime.sendMessage({
                type: 'EDITOR_LINES_DELETED',
                lineNumber: event.data.lineNumber,
                count: event.data.count,
                deleteType: event.data.deleteType || 'normal',
                mergeToLine: event.data.mergeToLine,
                source: event.data.source,  // 传递来源标记
                fileName: currentFileName   // 附带当前文件名，用于防止文件切换时的竞态条件
            }).catch((e) => {
                console.log('[Content] Failed to send EDITOR_LINES_DELETED:', e);
            });
        } else if (event.data && event.data.type === 'OVERLEAF_FILE_CHANGED') {
            // Forward file change to sidepanel
            console.log('[Content] Forwarding FILE_CHANGED:', event.data.fileName);
            currentFileName = event.data.fileName;
            chrome.runtime.sendMessage({
                type: 'FILE_CHANGED',
                fileName: event.data.fileName
            }).catch((e) => {
                console.log('[Content] Failed to send FILE_CHANGED:', e);
            });
        }
    });

    // Listen for messages from popup or background script
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
        if (request.type === 'GET_SELECTION') {
            // Get current selection from the page
            const selection = window.getSelection();
            const selectedText = selection.toString().trim();
            sendResponse({ selectedText: selectedText });
        } else if (request.type === 'GET_DOCUMENT_CONTENT') {
            // Function to get filename asynchronously
            const getFilenameAsync = () => {
                return new Promise((resolve) => {
                    if (currentFileName) {
                        resolve(currentFileName);
                        return;
                    }

                    // Request filename from injected script
                    const responseHandler = (event) => {
                        if (event.data && event.data.type === 'OVERLEAF_READER_FILENAME_RESPONSE') {
                            window.removeEventListener('message', responseHandler);
                            currentFileName = event.data.filename || '';
                            resolve(currentFileName);
                        }
                    };

                    window.addEventListener('message', responseHandler);
                    window.postMessage({ type: 'OVERLEAF_READER_GET_FILENAME' }, '*');

                    // Timeout after 1 second
                    setTimeout(() => {
                        window.removeEventListener('message', responseHandler);
                        resolve('');
                    }, 1000);
                });
            };

            if (documentContent) {
                getFilenameAsync().then(filename => {
                    sendResponse({
                        success: true,
                        content: documentContent,
                        length: documentContent.length,
                        filename: filename
                    });
                });
                return true; // Keep channel open for async response
            } else {
                // Try DOM inspection as immediate fallback
                tryDOMInspection();

                setTimeout(async () => {
                    const filename = await getFilenameAsync();
                    sendResponse({
                        success: !!documentContent,
                        content: documentContent || 'No content available yet. Please wait for the editor to load or try refreshing.',
                        length: documentContent ? documentContent.length : 0,
                        filename: filename
                    });
                }, 500);
                return true; // Keep channel open for async response
            }
        } else if (request.type === 'REFRESH_CONTENT') {
            // Re-inject script and try DOM inspection
            injectEditorAccessScript();
            tryDOMInspection();
            sendResponse({ success: true });
        } else if (request.type === 'GET_FRESH_DOCUMENT_CONTENT') {
            // 直接从 injected.js 获取最新内容（不使用缓存）
            console.log('[Content] Received GET_FRESH_DOCUMENT_CONTENT request');

            const responseHandler = (event) => {
                if (event.data && event.data.type === 'OVERLEAF_READER_FRESH_CONTENT_RESPONSE') {
                    window.removeEventListener('message', responseHandler);
                    console.log('[Content] Got fresh content response, length:', event.data.length);
                    // 同时更新缓存
                    if (event.data.content) {
                        documentContent = event.data.content;
                    }
                    sendResponse({
                        success: true,
                        content: event.data.content || '',
                        length: event.data.length || 0
                    });
                }
            };

            window.addEventListener('message', responseHandler);
            window.postMessage({ type: 'OVERLEAF_READER_GET_FRESH_CONTENT' }, '*');

            // Timeout after 2 seconds
            setTimeout(() => {
                window.removeEventListener('message', responseHandler);
                console.log('[Content] Fresh content request timed out, using cached content');
                sendResponse({
                    success: !!documentContent,
                    content: documentContent || '',
                    length: documentContent ? documentContent.length : 0
                });
            }, 2000);

            return true; // Keep channel open for async response
        } else if (request.type === 'GET_CURRENT_FILENAME') {
            // 直接从 injected.js 获取当前文件名（不使用缓存）
            console.log('[Content] Received GET_CURRENT_FILENAME request');

            const responseHandler = (event) => {
                if (event.data && event.data.type === 'OVERLEAF_READER_FILENAME_RESPONSE') {
                    window.removeEventListener('message', responseHandler);
                    console.log('[Content] Got filename response:', event.data.filename);
                    // 同时更新缓存
                    if (event.data.filename) {
                        currentFileName = event.data.filename;
                    }
                    sendResponse({
                        success: true,
                        filename: event.data.filename || ''
                    });
                }
            };

            window.addEventListener('message', responseHandler);
            window.postMessage({ type: 'OVERLEAF_READER_GET_FILENAME' }, '*');

            // Timeout after 500ms
            setTimeout(() => {
                window.removeEventListener('message', responseHandler);
                console.log('[Content] Filename request timed out, using cached filename');
                sendResponse({
                    success: !!currentFileName,
                    filename: currentFileName || ''
                });
            }, 500);

            return true; // Keep channel open for async response
        } else if (request.type === 'JUMP_TO_LINE') {
            // Try to jump to the specified line number
            const lineNumber = request.lineNumber;
            console.log('[Content] Received jump to line request:', lineNumber);

            // Use async approach for better handling
            setTimeout(() => {
                const success = jumpToLine(lineNumber);
                console.log('[Content] Jump result:', success);
                sendResponse({ success: success });
            }, 100);

            return true; // Keep channel open for async response
        } else if (request.type === 'UPDATE_CONTENT') {
            // Try to update the content in the editor
            const newContent = request.content;
            const originalContent = request.originalContent;
            console.log('[Content] Received update content request, length:', newContent.length);

            // Use async approach for better handling
            setTimeout(() => {
                const success = updateEditorContent(newContent, originalContent);
                console.log('[Content] Update result:', success);
                sendResponse({ success: success });
            }, 100);

            return true; // Keep channel open for async response
        } else if (request.type === 'DELETE_LINE') {
            // 精确删除指定行
            const lineNumber = request.lineNumber;
            const source = request.source;  // 来源标记
            console.log('[Content] Received delete line request:', lineNumber, 'source:', source);

            window.postMessage({
                type: 'OVERLEAF_READER_DELETE_LINE',
                lineNumber: lineNumber,
                source: source
            }, '*');

            const responseHandler = (event) => {
                if (event.data && event.data.type === 'OVERLEAF_READER_DELETE_LINE_RESULT') {
                    window.removeEventListener('message', responseHandler);
                    sendResponse({
                        success: event.data.success,
                        lineNumber: event.data.lineNumber
                    });
                }
            };
            window.addEventListener('message', responseHandler);

            setTimeout(() => {
                window.removeEventListener('message', responseHandler);
                sendResponse({ success: false, error: 'Timeout' });
            }, 5000);

            return true; // Keep channel open for async response
        } else if (request.type === 'MERGE_LINES') {
            // 合并行：将 fromLineNumber 行合并到 toLineNumber 行末尾
            const toLineNumber = request.toLineNumber;
            const fromLineNumber = request.fromLineNumber;
            const source = request.source;  // 来源标记
            console.log('[Content] Received merge lines request:', toLineNumber, '<-', fromLineNumber, 'source:', source);

            window.postMessage({
                type: 'OVERLEAF_READER_MERGE_LINES',
                toLineNumber: toLineNumber,
                fromLineNumber: fromLineNumber,
                source: source
            }, '*');

            const responseHandler = (event) => {
                if (event.data && event.data.type === 'OVERLEAF_READER_MERGE_LINES_RESULT') {
                    window.removeEventListener('message', responseHandler);
                    sendResponse({
                        success: event.data.success
                    });
                }
            };
            window.addEventListener('message', responseHandler);

            setTimeout(() => {
                window.removeEventListener('message', responseHandler);
                sendResponse({ success: false, error: 'Timeout' });
            }, 5000);

            return true; // Keep channel open for async response
        } else if (request.type === 'INSERT_LINE') {
            // 在指定行后插入新行
            const afterLineNumber = request.afterLineNumber;
            const source = request.source;  // 来源标记
            console.log('[Content] Received insert line request after:', afterLineNumber, 'source:', source);

            window.postMessage({
                type: 'OVERLEAF_READER_INSERT_LINE',
                afterLineNumber: afterLineNumber,
                source: source
            }, '*');

            const responseHandler = (event) => {
                if (event.data && event.data.type === 'OVERLEAF_READER_INSERT_LINE_RESULT') {
                    window.removeEventListener('message', responseHandler);
                    sendResponse({
                        success: event.data.success,
                        afterLineNumber: event.data.afterLineNumber
                    });
                }
            };
            window.addEventListener('message', responseHandler);

            setTimeout(() => {
                window.removeEventListener('message', responseHandler);
                sendResponse({ success: false, error: 'Timeout' });
            }, 5000);

            return true; // Keep channel open for async response
        } else if (request.type === 'UPDATE_LINE_CONTENT') {
            // 更新指定行的内容
            const lineNumber = request.lineNumber;
            const content = request.content;
            console.log('[Content] Received update line content request:', lineNumber, content.substring(0, 50));

            window.postMessage({
                type: 'OVERLEAF_READER_UPDATE_LINE_CONTENT',
                lineNumber: lineNumber,
                content: content
            }, '*');

            const responseHandler = (event) => {
                if (event.data && event.data.type === 'OVERLEAF_READER_UPDATE_LINE_CONTENT_RESULT') {
                    window.removeEventListener('message', responseHandler);
                    sendResponse({
                        success: event.data.success,
                        lineNumber: event.data.lineNumber
                    });
                }
            };
            window.addEventListener('message', responseHandler);

            setTimeout(() => {
                window.removeEventListener('message', responseHandler);
                sendResponse({ success: false, error: 'Timeout' });
            }, 5000);

            return true; // Keep channel open for async response
        } else if (request.type === 'ADD_CITATION_TO_BIB') {
            // Add citation to BIB file and insert reference
            const citationKey = request.citationKey;
            const bibEntry = request.bibEntry;
            console.log('[Content] Received add citation request:', citationKey);

            // Use async approach for better handling
            setTimeout(() => {
                window.postMessage({
                    type: 'OVERLEAF_READER_ADD_CITATION_TO_BIB',
                    citationKey: citationKey,
                    bibEntry: bibEntry
                }, '*');
            }, 100);

            // Listen for response from injected script
            const responseHandler = (event) => {
                if (event.data && event.data.type === 'OVERLEAF_READER_CITATION_RESPONSE') {
                    window.removeEventListener('message', responseHandler);
                    sendResponse({
                        success: event.data.success,
                        error: event.data.error
                    });
                }
            };
            window.addEventListener('message', responseHandler);

            // Timeout after 10 seconds
            setTimeout(() => {
                window.removeEventListener('message', responseHandler);
                sendResponse({
                    success: false,
                    error: 'Timeout waiting for citation operation'
                });
            }, 10000);

            return true; // Keep channel open for async response
        } else if (request.type === 'ADD_MULTIPLE_CITATIONS') {
            // Add multiple citations to BIB file and insert references
            const citationKeys = request.citationKeys;
            const bibEntries = request.bibEntries;
            console.log('[Content] Received add multiple citations request:', citationKeys);

            // Use async approach for better handling
            setTimeout(() => {
                window.postMessage({
                    type: 'OVERLEAF_READER_ADD_MULTIPLE_CITATIONS',
                    citationKeys: citationKeys,
                    bibEntries: bibEntries
                }, '*');
            }, 100);

            // Listen for response from injected script
            const responseHandler = (event) => {
                if (event.data && event.data.type === 'OVERLEAF_READER_CITATION_RESPONSE') {
                    window.removeEventListener('message', responseHandler);
                    sendResponse({
                        success: event.data.success,
                        error: event.data.error
                    });
                }
            };
            window.addEventListener('message', responseHandler);

            // Timeout after 20 seconds (longer for multiple citations)
            setTimeout(() => {
                window.removeEventListener('message', responseHandler);
                sendResponse({
                    success: false,
                    error: 'Timeout waiting for multiple citation operation'
                });
            }, 20000);

            return true; // Keep channel open for async response
        } else if (request.type === 'INSERT_CITATION') {
            // Insert citation reference only
            const citationKey = request.citationKey;
            console.log('[Content] Received insert citation request:', citationKey);

            // Use async approach for better handling
            setTimeout(() => {
                window.postMessage({
                    type: 'OVERLEAF_READER_INSERT_CITATION',
                    citationKey: citationKey
                }, '*');
            }, 100);

            // Listen for response from injected script
            const responseHandler = (event) => {
                if (event.data && event.data.type === 'OVERLEAF_READER_CITATION_RESPONSE') {
                    window.removeEventListener('message', responseHandler);
                    sendResponse({
                        success: event.data.success,
                        error: event.data.error
                    });
                }
            };
            window.addEventListener('message', responseHandler);

            // Timeout after 5 seconds
            setTimeout(() => {
                window.removeEventListener('message', responseHandler);
                sendResponse({
                    success: false,
                    error: 'Timeout waiting for citation insertion'
                });
            }, 5000);

            return true; // Keep channel open for async response
        } else if (request.type === 'REPLACE_SELECTED_TEXT') {
            // Replace selected text in the editor
            const originalText = request.originalText;
            const newText = request.newText;
            console.log('[Content] Received replace text request:', originalText, '->', newText);

            // Use async approach for better handling
            setTimeout(() => {
                window.postMessage({
                    type: 'OVERLEAF_READER_REPLACE_TEXT',
                    originalText: originalText,
                    newText: newText
                }, '*');
            }, 100);

            // Listen for response from injected script
            const responseHandler = (event) => {
                if (event.data && event.data.type === 'OVERLEAF_READER_REPLACE_RESPONSE') {
                    window.removeEventListener('message', responseHandler);
                    sendResponse({
                        success: event.data.success,
                        error: event.data.error
                    });
                }
            };
            window.addEventListener('message', responseHandler);

            // Timeout after 5 seconds
            setTimeout(() => {
                window.removeEventListener('message', responseHandler);
                sendResponse({
                    success: false,
                    error: 'Timeout waiting for text replacement'
                });
            }, 5000);

            return true; // Keep channel open for async response
        } else if (request.type === 'OPEN_COMPACT_VIEW') {
            // Open compact view panel with iframe
            const mode = request.mode || 'translation';
            console.log('[Content] Opening compact view with iframe, mode:', mode);
            const success = openCompactView(mode);
            sendResponse({ success: success });
            return true;
        } else if (request.type === 'CLOSE_COMPACT_VIEW') {
            // Close compact view panel
            console.log('[Content] Closing compact view');
            const success = closeCompactView();
            sendResponse({ success: success });
            return true;
        } else if (request.type === 'IS_COMPACT_VIEW_OPEN') {
            // Check if compact view is open
            sendResponse({ isOpen: isCompactViewOpen() });
            return true;
        } else if (request.type === 'GET_PROJECT_ZIP') {
            // Download project ZIP from Overleaf
            console.log('[Content] Received GET_PROJECT_ZIP request');

            // Extract project ID from current URL
            const urlMatch = window.location.href.match(/\/project\/([a-f0-9]+)/);
            if (!urlMatch) {
                sendResponse({ success: false, error: 'Cannot extract project ID from URL' });
                return true;
            }

            const projectId = urlMatch[1];
            const zipUrl = `/project/${projectId}/download/zip`;

            console.log('[Content] Downloading ZIP from:', zipUrl);

            // Fetch the ZIP file
            fetch(zipUrl, {
                method: 'GET',
                credentials: 'include'  // Include cookies for authentication
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.blob();
            })
            .then(blob => {
                // Convert blob to base64
                const reader = new FileReader();
                reader.onloadend = () => {
                    const base64data = reader.result.split(',')[1];
                    console.log('[Content] ZIP downloaded, size:', blob.size, 'bytes');
                    sendResponse({
                        success: true,
                        zipData: base64data,
                        projectId: projectId,
                        size: blob.size
                    });
                };
                reader.onerror = () => {
                    sendResponse({ success: false, error: 'Failed to read ZIP data' });
                };
                reader.readAsDataURL(blob);
            })
            .catch(error => {
                console.error('[Content] Error downloading ZIP:', error);
                sendResponse({ success: false, error: error.message });
            });

            return true; // Keep channel open for async response
        } else if (request.type === 'SHOW_DIFF_POPUP') {
            // Show diff preview popup in Overleaf editor
            const lineNumber = request.lineNumber;
            const originalText = request.originalText;
            const updatedText = request.updatedText;
            console.log('[Content] Received show diff popup request for line:', lineNumber);

            // Send message to injected script to show popup
            window.postMessage({
                type: 'OVERLEAF_READER_SHOW_DIFF_POPUP',
                lineNumber: lineNumber,
                originalText: originalText,
                updatedText: updatedText
            }, '*');

            // Listen for result from injected script
            const responseHandler = (event) => {
                if (event.data && event.data.type === 'OVERLEAF_DIFF_POPUP_RESULT') {
                    window.removeEventListener('message', responseHandler);
                    console.log('[Content] Diff popup result:', event.data);
                    sendResponse({
                        confirmed: event.data.confirmed,
                        success: event.data.success !== false,
                        error: event.data.error
                    });
                }
            };
            window.addEventListener('message', responseHandler);

            // Timeout after 60 seconds (user may take time to decide)
            setTimeout(() => {
                window.removeEventListener('message', responseHandler);
                sendResponse({
                    confirmed: false,
                    success: false,
                    error: 'Popup timed out'
                });
            }, 60000);

            return true; // Keep channel open for async response
        } else if (request.type === 'SHOW_EDIT_POPUP') {
            // Show edit confirmation popup in Overleaf editor (for ambient writing)
            const edit = request.edit;
            console.log('[Content] Received show edit popup request:', edit);

            // Send message to injected script to show popup
            window.postMessage({
                type: 'OVERLEAF_READER_SHOW_EDIT_POPUP',
                edit: edit
            }, '*');

            // Listen for result from injected script
            const editResponseHandler = (event) => {
                if (event.data && event.data.type === 'OVERLEAF_EDIT_POPUP_RESULT') {
                    window.removeEventListener('message', editResponseHandler);
                    console.log('[Content] Edit popup result:', event.data);
                    sendResponse({
                        confirmed: event.data.confirmed,
                        success: event.data.success !== false,
                        error: event.data.error
                    });
                }
            };
            window.addEventListener('message', editResponseHandler);

            // Timeout after 120 seconds (user may take time to review edits)
            setTimeout(() => {
                window.removeEventListener('message', editResponseHandler);
                sendResponse({
                    confirmed: false,
                    success: false,
                    error: 'Popup timed out'
                });
            }, 120000);

            return true; // Keep channel open for async response
        } else if (request.type === 'SIDEPANEL_STATE_CHANGED') {
            // Update floating button mini mode based on sidepanel state
            console.log('[Content] Sidepanel state changed:', request.isOpen);
            updateFloatingButtonMiniMode(request.isOpen);
            sendResponse({ success: true });
            return true;
        } else if (request.type === 'UPLOAD_IMAGE_TO_OVERLEAF') {
            // Upload generated image to Overleaf project
            const { base64Data, format, fileName } = request.data;
            console.log('[Content] Received upload image request:', {
                fileName,
                format,
                base64Length: base64Data ? base64Data.length : 0
            });

            // Extract project ID from current URL
            const urlMatch = window.location.href.match(/\/project\/([a-f0-9]+)/);
            if (!urlMatch) {
                sendResponse({ success: false, error: '无法获取项目ID' });
                return true;
            }

            const projectId = urlMatch[1];

            // Get CSRF token from page
            const csrfMeta = document.querySelector('meta[name="ol-csrfToken"]');
            const csrfToken = csrfMeta ? csrfMeta.content : '';

            if (!csrfToken) {
                console.warn('[Content] CSRF token not found, proceeding without it');
            }

            // Convert base64 to blob
            const byteCharacters = atob(base64Data);
            const byteNumbers = new Array(byteCharacters.length);
            for (let i = 0; i < byteCharacters.length; i++) {
                byteNumbers[i] = byteCharacters.charCodeAt(i);
            }
            const byteArray = new Uint8Array(byteNumbers);
            const mimeType = `image/${format}`;
            const blob = new Blob([byteArray], { type: mimeType });

            // Create FormData for upload
            const formData = new FormData();
            formData.append('qqfile', blob, fileName);

            // Get root folder ID from page meta tag
            const rootFolderMeta = document.querySelector('meta[name="ol-rootFolder"]');
            let rootFolderId = '';
            if (rootFolderMeta) {
                try {
                    const rootFolderData = JSON.parse(rootFolderMeta.content);
                    rootFolderId = rootFolderData._id || '';
                } catch (e) {
                    console.warn('[Content] Failed to parse root folder meta:', e);
                }
            }

            // Upload to Overleaf
            const uploadUrl = `/project/${projectId}/upload?folder_id=${rootFolderId || 'root'}`;

            console.log('[Content] Uploading to:', uploadUrl, 'size:', blob.size, 'rootFolderId:', rootFolderId);

            fetch(uploadUrl, {
                method: 'POST',
                credentials: 'include',
                headers: csrfToken ? { 'X-CSRF-Token': csrfToken } : {},
                body: formData
            })
            .then(response => {
                if (!response.ok) {
                    return response.text().then(text => {
                        throw new Error(`上传失败: ${response.status} - ${text}`);
                    });
                }
                return response.json();
            })
            .then(result => {
                console.log('[Content] Upload successful:', result);
                sendResponse({ success: true, result: result });
            })
            .catch(error => {
                console.error('[Content] Upload error:', error);
                sendResponse({ success: false, error: error.message });
            });

            return true; // Keep channel open for async response
        } else if (request.type === 'SHOW_LOADING_INDICATOR') {
            // 显示加载提示
            const taskName = request.taskName || '处理中';
            console.log('[Content] Showing loading indicator:', taskName);
            showLoadingIndicator(taskName);
            sendResponse({ success: true });
            return true;
        } else if (request.type === 'HIDE_LOADING_INDICATOR') {
            // 隐藏加载提示
            console.log('[Content] Hiding loading indicator');
            hideLoadingIndicator();
            sendResponse({ success: true });
            return true;
        } else if (request.type === 'SHOW_RESULT_POPUP') {
            // 显示结果弹窗
            const { title, message } = request;
            console.log('[Content] Showing result popup:', title);
            showResultPopup(title, message);
            sendResponse({ success: true });
            return true;
        }
        return true;
    });

    // =================== 加载提示和结果弹窗 ===================

    let loadingIndicatorElement = null;

    /**
     * 显示加载提示
     * @param {string} taskName - 任务名称
     */
    function showLoadingIndicator(taskName) {
        // 移除已有的提示
        hideLoadingIndicator();

        loadingIndicatorElement = document.createElement('div');
        loadingIndicatorElement.id = 'wenzhiyun-loading-indicator';
        loadingIndicatorElement.innerHTML = `
            <style>
                #wenzhiyun-loading-indicator {
                    position: fixed;
                    top: 20px;
                    right: 20px;
                    z-index: 999999;
                    background: linear-gradient(135deg, #7c3aed 0%, #5b21b6 100%);
                    color: white;
                    padding: 12px 20px;
                    border-radius: 12px;
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                    font-size: 14px;
                    font-weight: 500;
                    display: flex;
                    align-items: center;
                    gap: 12px;
                    box-shadow: 0 10px 40px rgba(124, 58, 237, 0.4);
                    animation: slideIn 0.3s ease-out;
                }
                @keyframes slideIn {
                    from {
                        transform: translateX(100px);
                        opacity: 0;
                    }
                    to {
                        transform: translateX(0);
                        opacity: 1;
                    }
                }
                .loading-spinner {
                    width: 20px;
                    height: 20px;
                    border: 2px solid rgba(255, 255, 255, 0.3);
                    border-top-color: white;
                    border-radius: 50%;
                    animation: spin 0.8s linear infinite;
                }
                @keyframes spin {
                    to { transform: rotate(360deg); }
                }
            </style>
            <div class="loading-spinner"></div>
            <span>${taskName}...</span>
        `;

        document.body.appendChild(loadingIndicatorElement);
    }

    /**
     * 隐藏加载提示
     */
    function hideLoadingIndicator() {
        if (loadingIndicatorElement) {
            loadingIndicatorElement.remove();
            loadingIndicatorElement = null;
        }
        // 也移除可能存在的旧元素
        const existing = document.getElementById('wenzhiyun-loading-indicator');
        if (existing) existing.remove();
    }

    /**
     * 显示结果弹窗
     * @param {string} title - 标题
     * @param {string} message - 消息内容
     */
    function showResultPopup(title, message) {
        // 移除已有的弹窗
        const existing = document.getElementById('wenzhiyun-result-popup');
        if (existing) existing.remove();

        const popup = document.createElement('div');
        popup.id = 'wenzhiyun-result-popup';
        popup.innerHTML = `
            <style>
                #wenzhiyun-result-popup {
                    position: fixed;
                    top: 50%;
                    left: 50%;
                    transform: translate(-50%, -50%);
                    z-index: 999999;
                    background: white;
                    border-radius: 16px;
                    box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
                    max-width: 500px;
                    width: 90%;
                    max-height: 80vh;
                    display: flex;
                    flex-direction: column;
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                    animation: popupFadeIn 0.3s ease-out;
                }
                @keyframes popupFadeIn {
                    from {
                        transform: translate(-50%, -50%) scale(0.9);
                        opacity: 0;
                    }
                    to {
                        transform: translate(-50%, -50%) scale(1);
                        opacity: 1;
                    }
                }
                #wenzhiyun-result-popup-overlay {
                    position: fixed;
                    inset: 0;
                    background: rgba(0, 0, 0, 0.5);
                    z-index: 999998;
                    animation: overlayFadeIn 0.2s ease-out;
                }
                @keyframes overlayFadeIn {
                    from { opacity: 0; }
                    to { opacity: 1; }
                }
                .result-popup-header {
                    display: flex;
                    align-items: center;
                    justify-content: space-between;
                    padding: 16px 20px;
                    background: linear-gradient(135deg, #7c3aed 0%, #5b21b6 100%);
                    border-radius: 16px 16px 0 0;
                    color: white;
                }
                .result-popup-header h3 {
                    margin: 0;
                    font-size: 16px;
                    font-weight: 600;
                }
                .result-popup-close {
                    background: rgba(255, 255, 255, 0.2);
                    border: none;
                    color: white;
                    width: 28px;
                    height: 28px;
                    border-radius: 8px;
                    cursor: pointer;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    transition: background 0.2s;
                }
                .result-popup-close:hover {
                    background: rgba(255, 255, 255, 0.3);
                }
                .result-popup-content {
                    padding: 20px;
                    overflow-y: auto;
                    max-height: 400px;
                    line-height: 1.6;
                    font-size: 14px;
                    color: #1e293b;
                    white-space: pre-wrap;
                    word-break: break-word;
                }
                .result-popup-footer {
                    padding: 12px 20px;
                    border-top: 1px solid #e2e8f0;
                    display: flex;
                    justify-content: flex-end;
                }
                .result-popup-btn {
                    padding: 8px 20px;
                    background: #7c3aed;
                    color: white;
                    border: none;
                    border-radius: 8px;
                    font-size: 14px;
                    font-weight: 500;
                    cursor: pointer;
                    transition: background 0.2s;
                }
                .result-popup-btn:hover {
                    background: #6d28d9;
                }
            </style>
            <div class="result-popup-header">
                <h3>${escapeHtml(title)}</h3>
                <button class="result-popup-close">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <line x1="18" y1="6" x2="6" y2="18"></line>
                        <line x1="6" y1="6" x2="18" y2="18"></line>
                    </svg>
                </button>
            </div>
            <div class="result-popup-content">${escapeHtml(message)}</div>
            <div class="result-popup-footer">
                <button class="result-popup-btn">确定</button>
            </div>
        `;

        // 创建遮罩层
        const overlay = document.createElement('div');
        overlay.id = 'wenzhiyun-result-popup-overlay';

        document.body.appendChild(overlay);
        document.body.appendChild(popup);

        // 绑定关闭事件
        const closePopup = () => {
            popup.remove();
            overlay.remove();
        };

        popup.querySelector('.result-popup-close').addEventListener('click', closePopup);
        popup.querySelector('.result-popup-btn').addEventListener('click', closePopup);
        overlay.addEventListener('click', closePopup);
    }

    /**
     * HTML 转义
     */
    function escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    // Observer for dynamic content changes
    function setupContentObserver() {
        const observer = new MutationObserver((mutations) => {
            let shouldCheck = false;
            for (const mutation of mutations) {
                if (mutation.type === 'childList' || mutation.type === 'characterData') {
                    shouldCheck = true;
                    break;
                }
            }

            if (shouldCheck && !documentContent) {
                // Debounce the check
                clearTimeout(setupContentObserver.timeout);
                setupContentObserver.timeout = setTimeout(() => {
                    tryDOMInspection();
                }, 1000);
            }
        });

        observer.observe(document.body, {
            childList: true,
            subtree: true,
            characterData: true
        });

        return observer;
    }

    // Initial setup
    console.log('[Overleaf Reader] Content script loaded');

    // Immediately try to inject script for faster content detection
    // Don't wait for 1 second - inject as soon as possible
    injectEditorAccessScript();

    // Track sidepanel state for mini mode
    let isSidepanelOpen = false;

    // Create floating button to open sidepanel
    function createFloatingButton() {
        // Check if button already exists
        if (document.getElementById('overleaf-helper-float-btn')) {
            return;
        }

        const floatBtn = document.createElement('div');
        floatBtn.id = 'overleaf-helper-float-btn';
        floatBtn.innerHTML = `
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M12 2L2 7L12 12L22 7L12 2Z" fill="currentColor"/>
                <path d="M2 17L12 22L22 17" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                <path d="M2 12L12 17L22 12" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
            </svg>
            <span>文智云</span>
        `;
        floatBtn.title = '点击打开Overleaf助手';

        // Add click event
        floatBtn.addEventListener('click', () => {
            chrome.runtime.sendMessage({
                type: 'OPEN_SIDEPANEL_FROM_BUTTON'
            }).catch(() => {
                console.log('[Overleaf Reader] Could not open sidepanel');
            });
        });

        document.body.appendChild(floatBtn);
        console.log('[Overleaf Reader] Floating button created');

        // Apply mini mode if sidepanel is already open
        if (isSidepanelOpen) {
            floatBtn.classList.add('mini');
        }
    }

    // Update floating button mini mode
    function updateFloatingButtonMiniMode(isOpen) {
        isSidepanelOpen = isOpen;
        const floatBtn = document.getElementById('overleaf-helper-float-btn');
        if (floatBtn) {
            if (isOpen) {
                floatBtn.classList.add('mini');
            } else {
                floatBtn.classList.remove('mini');
            }
        }
    }

    // Create the button after a short delay
    setTimeout(() => {
        createFloatingButton();
    }, 1000);

    // =================== 文智云按钮（工具栏） ===================

    /**
     * Inject the "文智云" button next to Overleaf's Recompile button
     * Clicking it opens the sidepanel (same as the floating button)
     */
    function injectWenzhiyunToolbarButton() {
        // Check if button already exists
        if (document.getElementById('wenzhiyun-toolbar-btn')) {
            return;
        }

        // Find the Recompile button group
        const compileButtonGroup = document.querySelector('.compile-button-group');
        if (!compileButtonGroup) {
            console.log('[Overleaf Reader] Recompile button group not found, will retry');
            return false;
        }

        // Create button container
        const wenzhiyunGroup = document.createElement('div');
        wenzhiyunGroup.id = 'wenzhiyun-toolbar-btn';
        wenzhiyunGroup.className = 'compile-button-group dropdown btn-group';
        wenzhiyunGroup.style.cssText = `
            border-radius: 12px;
            height: 28px;
            margin-left: 8px;
            background-color: #7c3aed;
        `;

        // Create the button
        const button = document.createElement('button');
        button.type = 'button';
        button.className = 'd-inline-grid align-items-center py-0 px-3 compile-button btn';
        button.style.cssText = `
            border-radius: 12px;
            font-size: 12px;
            background-color: #7c3aed;
            border-color: #7c3aed;
            color: white;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: background-color 0.2s;
        `;
        button.innerHTML = '<span class="button-content" aria-hidden="false">文智云</span>';
        button.title = '打开文智云助手侧边栏';

        // Add hover effect
        button.addEventListener('mouseenter', () => {
            button.style.backgroundColor = '#6d28d9';
        });
        button.addEventListener('mouseleave', () => {
            button.style.backgroundColor = '#7c3aed';
        });

        // Add click handler - open sidepanel
        button.addEventListener('click', () => {
            console.log('[Overleaf Reader] 文智云 toolbar button clicked');
            chrome.runtime.sendMessage({
                type: 'OPEN_SIDEPANEL_FROM_BUTTON'
            }).catch(() => {
                console.log('[Overleaf Reader] Could not open sidepanel');
            });
        });

        wenzhiyunGroup.appendChild(button);

        // Insert after the Recompile button group
        compileButtonGroup.parentNode.insertBefore(wenzhiyunGroup, compileButtonGroup.nextSibling);

        console.log('[Overleaf Reader] 文智云 toolbar button injected');
        return true;
    }

    // Try to inject button with retry logic
    function tryInjectToolbarButton() {
        if (!injectWenzhiyunToolbarButton()) {
            // Retry a few times if button not found
            let retries = 0;
            const maxRetries = 20;
            const retryInterval = setInterval(() => {
                retries++;
                if (injectWenzhiyunToolbarButton() || retries >= maxRetries) {
                    clearInterval(retryInterval);
                    if (retries >= maxRetries) {
                        console.log('[Overleaf Reader] Gave up injecting toolbar button after', maxRetries, 'retries');
                    }
                }
            }, 1000);
        }
    }

    // Inject button after page loads
    setTimeout(tryInjectToolbarButton, 2000);

    // Also use MutationObserver to inject when toolbar appears
    const toolbarObserver = new MutationObserver((mutations) => {
        for (const mutation of mutations) {
            if (mutation.addedNodes.length) {
                const compileGroup = document.querySelector('.compile-button-group');
                if (compileGroup && !document.getElementById('wenzhiyun-toolbar-btn')) {
                    injectWenzhiyunToolbarButton();
                }
            }
        }
    });
    toolbarObserver.observe(document.body, { childList: true, subtree: true });

    // =================== Compact View Functions ===================

    /**
     * Inject compact view CSS
     */
    function injectCompactViewStyles() {
        if (document.getElementById('overleaf-helper-compact-styles')) return;

        const link = document.createElement('link');
        link.id = 'overleaf-helper-compact-styles';
        link.rel = 'stylesheet';
        link.href = chrome.runtime.getURL('styles/compact-view.css');
        document.head.appendChild(link);
    }

    /**
     * Find the Overleaf editor container to embed compact view
     */
    function findEditorContainer() {
        // Try to find the main editor panel in Overleaf's new redesign
        // The editor is typically in a panel with data-panel attribute

        // First, look for the cm-editor and walk up to find its panel container
        const cmEditor = document.querySelector('.cm-editor');
        if (cmEditor) {
            let parent = cmEditor.parentElement;
            let depth = 0;
            while (parent && depth < 15) {
                // Look for panel containers
                if (parent.hasAttribute('data-panel') &&
                    !parent.classList.contains('ide-rail') &&
                    !parent.id?.includes('sidebar') &&
                    !parent.id?.includes('file-tree')) {
                    console.log('[Overleaf Reader] Found editor panel container:', parent.id || parent.className);
                    return parent;
                }
                parent = parent.parentElement;
                depth++;
            }
        }

        // Fallback: try specific selectors
        const selectors = [
            '[data-panel-id="panel-ide-editor"]',
            '.editor-container',
            '.source-editor-container'
        ];

        for (const selector of selectors) {
            const el = document.querySelector(selector);
            if (el) {
                console.log('[Overleaf Reader] Found editor container with selector:', selector);
                return el;
            }
        }

        console.log('[Overleaf Reader] Could not find editor container, using body fallback');
        return null;
    }

    /**
     * Create the compact view panel with iframe
     * @param {string} viewMode - 'translation' or 'ambient-writing'
     */
    function createCompactViewPanel(viewMode = 'translation') {
        const panel = document.createElement('div');
        panel.id = 'overleaf-helper-compact-panel';
        panel.style.height = compactPanelHeight;

        // Get sidepanel URL with compact mode parameter and view mode
        const sidepanelUrl = chrome.runtime.getURL('sidepanel/sidepanel.html') + `?mode=compact&view=${viewMode}`;

        // Create panel structure with iframe
        panel.innerHTML = `
            <div id="compact-panel-resizer"></div>
            <div id="compact-panel-overlay"></div>
            <button id="compact-panel-close" title="关闭简略视图">
                <svg viewBox="0 0 24 24">
                    <line x1="18" y1="6" x2="6" y2="18"></line>
                    <line x1="6" y1="6" x2="18" y2="18"></line>
                </svg>
            </button>
            <div id="compact-panel-iframe-container">
                <iframe id="compact-panel-iframe" src="${sidepanelUrl}"></iframe>
            </div>
        `;

        return panel;
    }

    /**
     * Set up compact panel event handlers
     */
    function setupCompactPanelEvents() {
        const resizer = document.getElementById('compact-panel-resizer');
        const overlay = document.getElementById('compact-panel-overlay');
        const closeBtn = document.getElementById('compact-panel-close');

        // Resize handle
        if (resizer) {
            resizer.addEventListener('mousedown', (e) => {
                e.preventDefault();
                isResizing = true;
                resizer.classList.add('dragging');
                if (overlay) overlay.classList.add('active');
                document.body.style.cursor = 'ns-resize';
                document.body.style.userSelect = 'none';

                const panel = document.getElementById('overleaf-helper-compact-panel');
                const startY = e.clientY;
                const startHeight = panel.offsetHeight;

                const onMouseMove = (moveEvent) => {
                    if (!isResizing) return;
                    const deltaY = startY - moveEvent.clientY;
                    const newHeight = Math.max(150, Math.min(window.innerHeight * 0.7, startHeight + deltaY));
                    panel.style.height = `${newHeight}px`;
                };

                const onMouseUp = () => {
                    isResizing = false;
                    resizer.classList.remove('dragging');
                    if (overlay) overlay.classList.remove('active');
                    document.body.style.cursor = '';
                    document.body.style.userSelect = '';
                    document.removeEventListener('mousemove', onMouseMove);
                    document.removeEventListener('mouseup', onMouseUp);

                    // Save height preference
                    const panel = document.getElementById('overleaf-helper-compact-panel');
                    if (panel) {
                        compactPanelHeight = panel.style.height;
                        localStorage.setItem('overleaf-helper-compact-height', compactPanelHeight);
                    }
                };

                document.addEventListener('mousemove', onMouseMove);
                document.addEventListener('mouseup', onMouseUp);
            });
        }

        // Close button
        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                closeCompactView();
                // Notify sidepanel that compact view is closed
                chrome.runtime.sendMessage({
                    type: 'COMPACT_VIEW_MESSAGE',
                    action: 'VIEW_CLOSED'
                });
            });
        }
    }

    // Store reference to original editor container and its parent
    let originalEditorContainer = null;
    let originalEditorParent = null;
    let wrapperElement = null;

    /**
     * Open compact view with iframe embedding sidepanel
     * Embeds the panel inside the Overleaf editor area
     * @param {string} viewMode - 'translation' or 'ambient-writing'
     */
    function openCompactView(viewMode = 'translation') {
        // Remove existing panel if any
        closeCompactView();

        // Inject styles
        injectCompactViewStyles();

        // Load saved height preference
        const savedHeight = localStorage.getItem('overleaf-helper-compact-height');
        if (savedHeight) {
            compactPanelHeight = savedHeight;
        }

        // Find the editor container
        const editorContainer = findEditorContainer();

        if (editorContainer) {
            // Save references for restore
            originalEditorContainer = editorContainer;
            originalEditorParent = editorContainer.parentElement;

            // Create wrapper
            wrapperElement = document.createElement('div');
            wrapperElement.id = 'overleaf-helper-compact-wrapper';

            // Create a container for the original editor content
            const editorWrapper = document.createElement('div');
            editorWrapper.className = 'overleaf-editor-original';

            // Move all children of editorContainer into editorWrapper
            while (editorContainer.firstChild) {
                editorWrapper.appendChild(editorContainer.firstChild);
            }

            // Create compact panel with viewMode
            compactViewPanel = createCompactViewPanel(viewMode);

            // Assemble: wrapper contains editorWrapper + compactPanel
            wrapperElement.appendChild(editorWrapper);
            wrapperElement.appendChild(compactViewPanel);

            // Put wrapper inside editor container
            editorContainer.appendChild(wrapperElement);

            // Make editor container flex
            editorContainer.style.display = 'flex';
            editorContainer.style.flexDirection = 'column';

            console.log('[Overleaf Reader] Compact view embedded in editor container, mode:', viewMode);
        } else {
            // Fallback: append to body with fixed positioning
            compactViewPanel = createCompactViewPanel(viewMode);
            compactViewPanel.style.position = 'fixed';
            compactViewPanel.style.left = '0';
            compactViewPanel.style.right = '0';
            compactViewPanel.style.bottom = '0';
            compactViewPanel.style.zIndex = '999998';
            compactViewPanel.style.height = compactPanelHeight;
            document.body.appendChild(compactViewPanel);

            console.log('[Overleaf Reader] Compact view opened with fallback (fixed position), mode:', viewMode);
        }

        // Setup events
        setupCompactPanelEvents();

        return true;
    }

    /**
     * Close compact view and restore original layout
     */
    function closeCompactView() {
        const panel = document.getElementById('overleaf-helper-compact-panel');
        const wrapper = document.getElementById('overleaf-helper-compact-wrapper');

        if (wrapper && originalEditorContainer) {
            // Restore original editor content
            const editorWrapper = wrapper.querySelector('.overleaf-editor-original');
            if (editorWrapper) {
                // Move children back to original container
                while (editorWrapper.firstChild) {
                    originalEditorContainer.appendChild(editorWrapper.firstChild);
                }
            }
            // Remove wrapper
            wrapper.remove();

            // Reset container styles
            originalEditorContainer.style.display = '';
            originalEditorContainer.style.flexDirection = '';
        } else if (panel) {
            // Fallback: just remove the panel
            panel.remove();
        }

        compactViewPanel = null;
        wrapperElement = null;

        console.log('[Overleaf Reader] Compact view closed');
        return true;
    }

    /**
     * Check if compact view is open
     */
    function isCompactViewOpen() {
        return !!document.getElementById('overleaf-helper-compact-panel');
    }

    // Wait for page to load, then set up observer
    setTimeout(() => {
        console.log('[Overleaf Reader] Setting up content observer...');
        setupContentObserver();
    }, 500);

    // Use faster retry for initial content detection
    // First few retries are fast (every 500ms), then slow down
    let retryCount = 0;
    const maxFastRetries = 20; // 10 seconds of fast retries

    const retryContentDetection = () => {
        if (!documentContent) {
            retryCount++;
            console.log(`[Overleaf Reader] No content yet, retry ${retryCount}...`);
            injectEditorAccessScript();
            tryDOMInspection();

            // Use 500ms for first 20 retries, then 3000ms
            const nextInterval = retryCount < maxFastRetries ? 500 : 3000;
            setTimeout(retryContentDetection, nextInterval);
        } else {
            console.log('[Overleaf Reader] Content found, stopping retries');
        }
    };

    // Start retry after 1 second
    setTimeout(retryContentDetection, 1000);

    // Clean up on page unload
    window.addEventListener('beforeunload', () => {
        // Cleanup if needed
    });

    // =================== 氛围写作功能支持 ===================
    // 监听来自 injected.js 的消息并转发给 sidepanel

    /**
     * 设置氛围写作消息监听
     */
    /**
     * 先唤醒侧边栏（如果没有缩略视图），然后发送消息
     * @param {object} message - 要发送的消息
     * @param {number} delay - 延迟时间（毫秒），默认 300
     */
    function openSidepanelAndSendMessage(message, delay = 300) {
        // 检查是否已有缩略视图打开
        const hasCompactView = isCompactViewOpen();

        if (hasCompactView) {
            // 如果有缩略视图，直接发送消息（缩略视图内的 iframe 会接收）
            console.log('[Overleaf Reader] Compact view is open, sending message directly');
            chrome.runtime.sendMessage(message).catch((error) => {
                console.error('[Overleaf Reader] Error sending message:', error);
            });
        } else {
            // 没有缩略视图，先唤醒侧边栏
            chrome.runtime.sendMessage({
                type: 'OPEN_SIDEPANEL_FROM_BUTTON'
            }).then(() => {
                console.log('[Overleaf Reader] Sidepanel open request sent');
            }).catch(() => {
                // 忽略错误
            });

            // 延迟发送消息，等待侧边栏打开
            setTimeout(() => {
                chrome.runtime.sendMessage(message).catch((error) => {
                    console.error('[Overleaf Reader] Error sending message after sidepanel open:', error);
                });
            }, delay);
        }
    }

    function setupAmbientWritingSupport() {
        console.log('[Overleaf Reader] Setting up ambient writing support');

        // 监听来自 injected.js 的消息
        window.addEventListener('message', (event) => {
            // 插入氛围写作
            if (event.data && event.data.type === 'OVERLEAF_INSERT_TO_AMBIENT_WRITING') {
                console.log('[Overleaf Reader] Received ambient writing insert request:', event.data.data);

                // 先唤醒侧边栏，然后转发给 sidepanel
                openSidepanelAndSendMessage({
                    type: 'INSERT_TO_AMBIENT_WRITING',
                    data: event.data.data
                });
            }

            // 监听文件树右键菜单的"插入到氛围写作"
            if (event.data && event.data.type === 'OVERLEAF_INSERT_FILE_TO_AMBIENT_WRITING') {
                console.log('[Overleaf Reader] Received file insert to ambient writing:', event.data.data);

                // 先唤醒侧边栏，然后转发给 sidepanel
                openSidepanelAndSendMessage({
                    type: 'INSERT_FILE_TO_AMBIENT_WRITING',
                    data: event.data.data
                });
            }

            // 监听自动引用按钮点击消息
            if (event.data && event.data.type === 'OVERLEAF_AUTO_CITE') {
                console.log('[Overleaf Reader] Received auto-cite request:', event.data.selectedText);

                // 先唤醒侧边栏，然后转发给 sidepanel
                openSidepanelAndSendMessage({
                    type: 'AUTO_CITE_SHORTCUT',
                    selectedText: event.data.selectedText
                });
            }

            // 监听文智云修复编译错误请求
            if (event.data && event.data.type === 'OVERLEAF_WENZHIYUN_FIX_ERROR') {
                console.log('[Overleaf Reader] Received 文智云修复 request:', event.data.data);

                // 先尝试唤醒侧边栏，然后发送修复请求
                chrome.runtime.sendMessage({
                    type: 'OPEN_SIDEPANEL_FROM_BUTTON'
                }).then(() => {
                    console.log('[Overleaf Reader] Sidepanel open request sent');
                }).catch(() => {
                    // 忽略错误
                });

                // 延迟发送修复请求，等待侧边栏打开
                setTimeout(() => {
                    chrome.runtime.sendMessage({
                        type: 'WENZHIYUN_FIX_COMPILE_ERROR',
                        data: event.data.data
                    }).then(response => {
                        console.log('[Overleaf Reader] 文智云修复 response:', response);
                        // 将结果发回 injected.js
                        window.postMessage({
                            type: 'OVERLEAF_WENZHIYUN_FIX_RESULT',
                            success: response?.success || false,
                            error: response?.error
                        }, '*');
                    }).catch((error) => {
                        console.error('[Overleaf Reader] 文智云修复 error:', error);
                        // 将错误发回 injected.js
                        window.postMessage({
                            type: 'OVERLEAF_WENZHIYUN_FIX_RESULT',
                            success: false,
                            error: error.message || '请打开侧边栏后重试'
                        }, '*');
                    });
                }, 500);
            }

            // 监听语法检查请求
            if (event.data && event.data.type === 'OVERLEAF_GRAMMAR_CHECK') {
                console.log('[Overleaf Reader] Received grammar check request:', event.data.data);

                // 先唤醒侧边栏，然后转发给 sidepanel
                openSidepanelAndSendMessage({
                    type: 'GRAMMAR_CHECK',
                    data: event.data.data
                });
            }

            // 监听一键润色请求
            if (event.data && event.data.type === 'OVERLEAF_POLISH') {
                console.log('[Overleaf Reader] Received polish request:', event.data.data);

                // 先唤醒侧边栏，然后转发给 sidepanel
                openSidepanelAndSendMessage({
                    type: 'POLISH',
                    data: event.data.data
                });
            }

            // =================== 自动续写功能 ===================

            // 监听续写请求
            if (event.data && event.data.type === 'OVERLEAF_CONTINUE_WRITING') {
                console.log('[Overleaf Reader] Received continue writing request, content length:', event.data.contentBeforeCursor?.length);

                // 转发给 sidepanel
                chrome.runtime.sendMessage({
                    type: 'CONTINUE_WRITING_REQUEST',
                    contentBeforeCursor: event.data.contentBeforeCursor
                }).then(response => {
                    console.log('[Overleaf Reader] Continue writing response:', response);
                    // 将结果发回 injected.js
                    window.postMessage({
                        type: 'OVERLEAF_CONTINUE_WRITING_RESULT',
                        success: response?.success || false,
                        continuation: response?.continuation,
                        error: response?.error
                    }, '*');
                }).catch((error) => {
                    console.error('[Overleaf Reader] Continue writing error:', error);
                    // 将错误发回 injected.js
                    window.postMessage({
                        type: 'OVERLEAF_CONTINUE_WRITING_RESULT',
                        success: false,
                        error: error.message || '续写请求失败，请确保侧边栏已打开'
                    }, '*');
                });
            }

            // 监听获取续写开关状态请求
            if (event.data && event.data.type === 'OVERLEAF_GET_CONTINUE_WRITING_ENABLED') {
                console.log('[Overleaf Reader] Received get continue writing enabled request');

                // 从 sidepanel 获取状态
                chrome.runtime.sendMessage({
                    type: 'GET_CONTINUE_WRITING_ENABLED'
                }).then(response => {
                    console.log('[Overleaf Reader] Continue writing enabled response:', response);
                    // 将状态发回 injected.js
                    window.postMessage({
                        type: 'OVERLEAF_CONTINUE_WRITING_ENABLED_CHANGED',
                        enabled: response?.enabled || false
                    }, '*');
                }).catch((error) => {
                    console.error('[Overleaf Reader] Get continue writing enabled error:', error);
                    // 默认关闭
                    window.postMessage({
                        type: 'OVERLEAF_CONTINUE_WRITING_ENABLED_CHANGED',
                        enabled: false
                    }, '*');
                });
            }
        });
    }

    // 监听来自 sidepanel 的续写开关状态变化
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
        if (request.type === 'CONTINUE_WRITING_ENABLED_CHANGED') {
            console.log('[Overleaf Reader] Continue writing enabled changed from sidepanel:', request.enabled);
            // 转发给 injected.js
            window.postMessage({
                type: 'OVERLEAF_CONTINUE_WRITING_ENABLED_CHANGED',
                enabled: request.enabled
            }, '*');
            sendResponse({ success: true });
            return true;
        }
    });

    // 设置拖拽支持
    setupAmbientWritingSupport();

    // =================== 首次使用教程（全页面弹窗） ===================

    const TUTORIAL_IMAGES = [
        { src: 'assets/tutorial/vibe_write1.png', title: '氛围写作 - AI智能续写', desc: '像使用Cursor写代码一样写论文，输入需求后AI自动理解上下文并生成高质量学术内容' },
        { src: 'assets/tutorial/vibe_figure.png', title: '氛围写作 - 图像理解', desc: '上传图片后AI自动理解图像内容，智能生成Figure Caption和相关描述文字' },
        { src: 'assets/tutorial/vibe_code.png', title: '氛围写作 - 代码解读', desc: '读取训练代码自动理解算法细节，帮你撰写方法描述和实验设置章节' },
        { src: 'assets/tutorial/continue.png', title: '智能续写', desc: '根据论文内容实时推荐续写内容，AI理解上下文自动补全段落，让写作更流畅' },
        { src: 'assets/tutorial/draw.png', title: 'AI绘图', desc: '调用Nano Banana Pro模型进行论文图像的智能绘制与修改，轻松创建专业学术图表' },
        { src: 'assets/tutorial/zh_update1.png', title: '边写边译', desc: '用中文思维写英文论文，在中文文档修改后实时同步到英文文档，保持LaTeX格式不变' },
        { src: 'assets/tutorial/pdf.png', title: 'PDF编译', desc: '突破Overleaf非会员10秒编译限制，无论文档多长都能顺利编译生成PDF' },
        { src: 'assets/tutorial/gramma.png', title: '语法校对', desc: '智能检测语法、拼写、时态等问题，提供专业的学术英语表达建议，一键修复' },
        { src: 'assets/tutorial/error.png', title: '错误修复', desc: '智能分析LaTeX编译错误和警告信息，快速定位问题并提供一键修复方案' },
        { src: 'assets/tutorial/cite1.png', title: '文献引用', desc: '根据上下文智能推荐相关文献，自动搜索并生成BibTeX格式的引用条目' }
    ];

    let tutorialCurrentStep = 0;
    let tutorialModal = null;

    /**
     * 检查是否需要显示教程
     */
    async function checkAndShowTutorial() {
        try {
            const result = await chrome.storage.local.get(['tutorialShown']);
            if (!result.tutorialShown) {
                // 延迟显示，等待页面加载完成
                setTimeout(() => {
                    showTutorialModal();
                }, 2000);
            }
        } catch (error) {
            console.error('[Overleaf Reader] Error checking tutorial status:', error);
        }
    }

    /**
     * 显示教程弹窗
     */
    function showTutorialModal() {
        if (tutorialModal) return;

        tutorialCurrentStep = 0;

        // 创建弹窗容器
        tutorialModal = document.createElement('div');
        tutorialModal.id = 'wenzhiyun-tutorial-modal';
        tutorialModal.innerHTML = `
            <style>
                #wenzhiyun-tutorial-modal {
                    position: fixed;
                    inset: 0;
                    z-index: 999999;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    background: rgba(0, 0, 0, 0.85);
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                }
                .tutorial-container {
                    background: white;
                    border-radius: 16px;
                    width: 90%;
                    max-width: 900px;
                    max-height: 90vh;
                    display: flex;
                    flex-direction: column;
                    overflow: hidden;
                    box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.5);
                }
                .tutorial-header {
                    display: flex;
                    align-items: center;
                    justify-content: space-between;
                    padding: 10px 20px;
                    background: linear-gradient(135deg, #16a34a 0%, #15803d 100%);
                    color: white;
                }
                .tutorial-header h2 {
                    margin: 0;
                    font-size: 15px;
                    font-weight: 600;
                    display: flex;
                    align-items: center;
                    gap: 8px;
                }
                .tutorial-header h2 svg {
                    width: 18px;
                    height: 18px;
                }
                .tutorial-close-btn {
                    background: rgba(255,255,255,0.2);
                    border: none;
                    color: white;
                    width: 26px;
                    height: 26px;
                    border-radius: 6px;
                    cursor: pointer;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    transition: background 0.2s;
                }
                .tutorial-close-btn:hover {
                    background: rgba(255,255,255,0.3);
                }
                .tutorial-dots {
                    display: flex;
                    justify-content: center;
                    gap: 6px;
                    padding: 8px;
                    background: #f8fafc;
                    border-bottom: 1px solid #e2e8f0;
                }
                .tutorial-dot {
                    width: 8px;
                    height: 8px;
                    border-radius: 50%;
                    background: #cbd5e1;
                    transition: all 0.3s;
                    cursor: pointer;
                }
                .tutorial-dot.active {
                    background: #16a34a;
                    transform: scale(1.3);
                }
                .tutorial-content {
                    flex: 1;
                    display: flex;
                    flex-direction: column;
                    min-height: 0;
                    overflow: hidden;
                }
                .tutorial-image-container {
                    flex: 1;
                    min-height: 0;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    padding: 16px 24px;
                    background: #f8fafc;
                }
                .tutorial-image {
                    max-width: 100%;
                    max-height: 100%;
                    object-fit: contain;
                    border-radius: 12px;
                    border: 1px solid #e2e8f0;
                    box-shadow: 0 8px 24px rgba(0, 0, 0, 0.12);
                }
                .tutorial-text {
                    padding: 16px 24px;
                    text-align: center;
                    flex-shrink: 0;
                    background: white;
                    border-top: 1px solid #e2e8f0;
                }
                .tutorial-text h3 {
                    margin: 0 0 6px 0;
                    font-size: 18px;
                    font-weight: 700;
                    color: #1e293b;
                }
                .tutorial-text p {
                    margin: 0;
                    font-size: 14px;
                    color: #64748b;
                    line-height: 1.5;
                }
                .tutorial-footer {
                    display: flex;
                    align-items: center;
                    justify-content: space-between;
                    padding: 12px 24px;
                    background: #f8fafc;
                    border-top: 1px solid #e2e8f0;
                }
                .tutorial-btn {
                    padding: 10px 20px;
                    border-radius: 8px;
                    font-size: 14px;
                    font-weight: 500;
                    cursor: pointer;
                    transition: all 0.2s;
                    border: none;
                }
                .tutorial-btn-prev {
                    background: #e2e8f0;
                    color: #475569;
                }
                .tutorial-btn-prev:hover:not(:disabled) {
                    background: #cbd5e1;
                }
                .tutorial-btn-prev:disabled {
                    opacity: 0.5;
                    cursor: not-allowed;
                }
                .tutorial-btn-next {
                    background: #16a34a;
                    color: white;
                }
                .tutorial-btn-next:hover {
                    background: #15803d;
                }
                .tutorial-step-text {
                    font-size: 14px;
                    color: #94a3b8;
                }
            </style>
            <div class="tutorial-container">
                <div class="tutorial-header">
                    <h2>
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="12" cy="12" r="10"></circle>
                            <path d="M12 16v-4M12 8h.01"></path>
                        </svg>
                        欢迎使用文智云助手
                    </h2>
                    <button class="tutorial-close-btn" id="tutorial-close">
                        <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <line x1="18" y1="6" x2="6" y2="18"></line>
                            <line x1="6" y1="6" x2="18" y2="18"></line>
                        </svg>
                    </button>
                </div>
                <div class="tutorial-dots" id="tutorial-dots"></div>
                <div class="tutorial-content">
                    <div class="tutorial-image-container">
                        <img class="tutorial-image" id="tutorial-img" src="" alt="">
                    </div>
                    <div class="tutorial-text">
                        <h3 id="tutorial-title"></h3>
                        <p id="tutorial-desc"></p>
                    </div>
                </div>
                <div class="tutorial-footer">
                    <button class="tutorial-btn tutorial-btn-prev" id="tutorial-prev">上一步</button>
                    <span class="tutorial-step-text" id="tutorial-step-text"></span>
                    <button class="tutorial-btn tutorial-btn-next" id="tutorial-next">下一步</button>
                </div>
            </div>
        `;

        document.body.appendChild(tutorialModal);

        // 创建步骤圆点
        const dotsContainer = document.getElementById('tutorial-dots');
        TUTORIAL_IMAGES.forEach((_, i) => {
            const dot = document.createElement('span');
            dot.className = 'tutorial-dot' + (i === 0 ? ' active' : '');
            dot.addEventListener('click', () => goToTutorialStep(i));
            dotsContainer.appendChild(dot);
        });

        // 绑定事件
        document.getElementById('tutorial-close').addEventListener('click', closeTutorialModal);
        document.getElementById('tutorial-prev').addEventListener('click', () => goToTutorialStep(tutorialCurrentStep - 1));
        document.getElementById('tutorial-next').addEventListener('click', () => {
            if (tutorialCurrentStep < TUTORIAL_IMAGES.length - 1) {
                goToTutorialStep(tutorialCurrentStep + 1);
            } else {
                closeTutorialModal();
            }
        });

        // 显示第一步
        updateTutorialUI();

        console.log('[Overleaf Reader] Tutorial modal shown');
    }

    /**
     * 跳转到指定步骤
     */
    function goToTutorialStep(step) {
        if (step < 0 || step >= TUTORIAL_IMAGES.length) return;
        tutorialCurrentStep = step;
        updateTutorialUI();
    }

    /**
     * 更新教程 UI
     */
    function updateTutorialUI() {
        const data = TUTORIAL_IMAGES[tutorialCurrentStep];

        // 更新图片
        const img = document.getElementById('tutorial-img');
        img.src = chrome.runtime.getURL(data.src);
        img.alt = data.title;

        // 更新文字
        document.getElementById('tutorial-title').textContent = data.title;
        document.getElementById('tutorial-desc').textContent = data.desc;

        // 更新步骤文本
        document.getElementById('tutorial-step-text').textContent = `${tutorialCurrentStep + 1} / ${TUTORIAL_IMAGES.length}`;

        // 更新圆点
        const dots = document.querySelectorAll('.tutorial-dot');
        dots.forEach((dot, i) => {
            dot.classList.toggle('active', i === tutorialCurrentStep);
        });

        // 更新按钮
        document.getElementById('tutorial-prev').disabled = tutorialCurrentStep === 0;
        document.getElementById('tutorial-next').textContent =
            tutorialCurrentStep === TUTORIAL_IMAGES.length - 1 ? '开始使用' : '下一步';
    }

    /**
     * 关闭教程弹窗
     */
    function closeTutorialModal() {
        if (tutorialModal) {
            tutorialModal.remove();
            tutorialModal = null;
        }

        // 标记教程已显示
        chrome.storage.local.set({ tutorialShown: true }).then(() => {
            console.log('[Overleaf Reader] Tutorial marked as shown');
        }).catch(err => {
            console.error('[Overleaf Reader] Error marking tutorial as shown:', err);
        });
    }

    // 检查并显示教程
    checkAndShowTutorial();

})();